/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB. All rights reserved.
 
   This program is an unpublished copyrighted work which is proprietary
   to HID Global. This computer program includes Confidential,
   Proprietary Information and is a Trade Secret of HID Global.
   Any use, disclosure, modification and/or reproduction is prohibited
   unless authorized in writing by HID Global.

   WARNING:  Unauthorized reproduction of this program as well as
   unauthorized preparation of derivative works based upon the
   program or distribution of copies by sale, rental, lease or
   lending are violations of federal copyright laws and state trade
   secret laws, punishable by civil and criminal penalties.
--------------------------------------------------------------------------- */

/*
 *BSIf.h
 */

#ifndef __BSIF_H__
#define __BSIF_H__

#ifdef __cplusplus
extern "C" {
#endif

#define IN
#define OUT
#define INOUT

/*
Standard includes and definitions
*/
/*
Windows
*/
#if defined(WIN32) || defined(WIN16)
#include <windows.h>
/*
Unix
*/
#else
#include <stdio.h>
#define CALLBACK
#define TRUE 1
#define FALSE 0
#endif

/*
 *
 * gscBsiUtilAcquireContext()
 *
 * Purpose: This function shall establish a session with the smart card by submitting
 *			the appropriate Authenticator in the BSIAuthenticator structure.  For ACRs 
 *			requiring external authentication (XAUTH), the uszAuthValue field of the 
 *			BSIAuthenticator structure must contain a cryptogram calculated by encrypting 
 *			a random challenge. 
 *
 * Returns:	BSI_OK
 *			BSI_BAD_HANDLE 
 *			BSI_BAD_AID 
 *			BSI_ACR_NOT_AVAILABLE 
 *			BSI_BAD_AUTH 
 *			BSI_CARD_REMOVED 
 *			BSI_PIN_LOCKED 
 *			BSI_UNKNOWN_ERROR
 *
 */
BSI_RETURN CALLBACK gscBsiUtilAcquireContext( 
	IN 		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect()			*/
	IN 		BSI_BYTE_PTR	uszAID,					/* Target container AID value.								*/
	IN 		BSI_ULONG		unAIDLen,				/* AID value length in bytes								*/
	IN 		BSI_AUTH_PTR	strctAuthenticator,		/* An array of structures containing the authenticator(s) 
														specified by the ACR									*/
	IN 		BSI_ULONG		unAuthNb				/* Number of authenticator structures contained in 
														strctAuthenticator.										*/
); 


/*
 * gscBsiUtilConnect()
 *
 * Purpose: Establish a logical connection with the card in a specified reader. 
 *
 * Returns: BSI_OK
 *			BSI_BAD_PARAM 
 *			BSI_UNKNOWN_READER 
 *			BSI_CARD_ABSENT
 *			BSI_TIMEOUT_ERROR
 *			BSI_UNKNOWN_ERROR   
 *
 */

BSI_RETURN CALLBACK gscBsiUtilConnect(
	IN		BSI_BYTE_PTR		uszReaderName,		/* Name of the reader that the card is inserted into.		*/
	IN		BSI_ULONG			unReaderNameLen,	/* Length of the reader name in bytes						*/
	OUT		BSI_CARD_HANDLE_PTR	hCard				/* Card connection handle.									*/
);


/*
 * gscBsiUtilDisconnect() 
 *
 * Purpose: Terminate a logical connection to a card. 
 *
 * Returns: BSI_OK 
 *			BSI_BAD_HANDLE 
 *			BSI_CARD_REMOVED 
 *			BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscBsiUtilDisconnect( 
	IN		BSI_CARD_HANDLE		hCard				/* Card connection handle from gscBsiUtilConnect()			*/
); 


/*
 * gscBsiUtilBeginTransaction() 
 *
 * Purpose: Starts an exclusive transaction with the smart card referenced by hCard. When the
 * transaction starts, all other applications are precluded from accessing the smart card
 * while the transaction is in progress. Two types of calls can be made with that
 * function: a blocking transaction call and a non-blocking transaction call. A boolean
 * type parameter identify which mode is called. In the non-blocking mode, the call
 * will return immediately if another client has an active transaction lock. The returned
 * error code will be BSI_SC_LOCKED . In the blocking mode, the call will wait
 * indefinitely for any active transaction locks to be released. A transaction must be
 * completed by a call to gscBsiUtilEndTransaction().
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE			
 *          BSI_UNKNOWN_ERROR 
 *			BSI_SC_LOCKED
 *			BSI_NOT_TRANSACTED
 *			BSI_NO_SPSSERVICE
 */
BSI_RETURN CALLBACK gscBsiUtilBeginTransaction( 
	IN		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect()			*/
	IN		BOOL blType								/* Boolean specifying the type of transaction call
													   ( blTypeset to "true" in blocking mode.
													   blType set to "false" in non-blocking mode)				*/
);

/*
 * gscBsiUtilEndTransaction() 
 *
 * Purpose: Completes a previously started transaction, allowing other applications to resume interactions with the card.
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE			
 *          BSI_UNKNOWN_ERROR
 *			BSI_NOT_TRANSACTED
 *			BSI_NO_SPSSERVICE
 */
BSI_RETURN CALLBACK gscBsiUtilEndTransaction( 
	IN		BSI_CARD_HANDLE	hCard					/* Card connection handle from gscBsiUtilConnect()			*/
);


/*
 * gscBsiUtilGetVersion() 
 * 
 * Purpose: Returns the BSI implementation version. 
 *
 * Returns: BSI_OK 
 *			BSI_BAD_PARAM 
 *			BSI_INSUFFICIENT_BUFFER 
 *			BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscBsiUtilGetVersion (
	IN OUT	BSI_BYTE_PTR		uszVersion,			/* The BSI and SCSPM version formatted as .major,minor,
													revision, build_number\0									*/
	IN OUT	BSI_ULONG_PTR		unVersionLen		/* Length of the version string								*/
); 
 

/*
 * gscBsiUtilGetCardProperties() 
 *
 * Purpose: Retrieves ID and capability information for the card.
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_CARD_REMOVED
 *          BSI_SC_LOCKED
 *          BSI_BAD_PARAM 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_NO_CARDSERVICE 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiUtilGetCardProperties( 
	IN		BSI_CARD_HANDLE		hCard,					/* Card connection handle from gscBsiUtilConnect().			*/
	IN OUT	BSI_BYTE_PTR		uszCCCUniqueID,			/* Buffer for the Card Capability Container ID, represented 
														   in ASCII Hexadecimal.									*/
	IN OUT	BSI_ULONG_PTR		punCCCUniqueIDLen,		/* Length of the CCC Unique ID string (input).  Length of 
														the returned Card Unique ID string including the 
														null terminator (output).									*/
	OUT		BSI_ULONG_PTR		punCardCapability		/* Bit mask value defining the providers supported by the 
														card.														*/
); 


/*
 * gscBsiUtilGetCardStatus() 
 * Purpose: Checks whether a given card handle is associated with a card 
 *			that is inserted into a powered up reader.
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_CARD_REMOVED 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiUtilGetCardStatus( 
	IN BSI_CARD_HANDLE	hCard					/* Card connection handle from gscBsiUtilConnect().			*/
); 
 

/*
 * gscBsiUtilGetExtendedErrorText() 
 *
 * Purpose: When a BSI function call returns an error, an application can 
 *			make a subsequent call to this function to receive additional 
 *			implementation specific error information, if available. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_BAD_PARAM 
 *          BSI_NO_TEXT_AVAILABLE 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiUtilGetExtendedErrorText( 
	IN		BSI_CARD_HANDLE	hCard,					/* Card connection handle gscBsiUtilConnect()				*/
	OUT		BSI_CHAR		uszErrorText[BSI_ERROR_TEXT_LEN]
												/* buffer containing an implementation specific error text 
													string													*/
); 
 

/*
 * gscBsiUtilGetReaderList() 
 *
 * Purpose: Retrieves the list of available readers. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_PARAM 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiUtilGetReaderList( 
	IN OUT	BSI_BYTE_PTR		uszReaderList,		/* Reader list buffer.  The reader list is returned as a 
													multi-string, each reader name terminated by a .\0..  
													The list itself is terminated by an additional trailing 
													.\0. character.											*/
	IN OUT	BSI_ULONG_PTR		unReaderListLen		/* Reader list length in bytes including all terminating 
													.\0. characters											*/
); 


/*
 * gscBsiUtilPassthru() 
 *
 * Purpose: Allows a client application to send a .raw. APDU through the 
 *			BSI directly to the card and receive the APDU-level response. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_BAD_PARAM 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_CARD_REMOVED
 *			BSI_SC_LOCKED
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiUtilPassthru( 
	IN BSI_CARD_HANDLE		hCard,				/* Card connection handle from gscBsiUtilConnect().			*/
	IN BSI_BYTE_PTR			uszCardCommand,		/* The APDU to be sent to the card							*/
	IN BSI_ULONG			unCardCommandLen,	/* Length of the APDU string to be sent						*/
	IN OUT BSI_BYTE_PTR		uszCardResponse,	/* Pre-allocated buffer for the APDU response from the card. 
													The response must include the status bytes SW1 and SW2 
													returned by the card.  If the size of the buffer is 
													insufficient, the SPS shall return truncated response 
													data and the return code BSI_INSUFFICIENT_BUFFER. 
													That parameter must  be in ASCII hexadecimal format.	*/
	IN OUT BSI_ULONG_PTR	punCardResponseLen			/* Length of the APDU response.  If the size of the 
													uszCardResponse buffer is insufficient, the SPS shall 
													return the correct size in this field.					*/
); 
 

/*
 * gscBsiUtilReleaseContext() 
 *
 * Purpose: Terminate a session with the card. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED
 *			BSI_SC_LOCKED
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiUtilReleaseContext( 
	IN		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect()									*/
	IN		BSI_BYTE_PTR	uszAID,					/* Target  container AID value. The parameter shall be in ASCII hexadecimal format.	*/
	IN		BSI_ULONG		unAIDLen				/* AID value length in bytes														*/
);


/*
 * gscBsiGcDataCreate() 
 *
 * Purpose: Create a new data item in {Tag, Length, Value} format in the selected container. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_BAD_AID
 *			BSI_SC_LOCKED
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_NO_MORE_SPACE 
 *          BSI_TAG_EXISTS 
 *			BSI_IO_ERROR 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGcDataCreate( 
	IN 		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect()									*/
 	IN 		BSI_BYTE_PTR	uszAID,					/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN 		BSI_ULONG		unAIDLen,				/* AID value length in bytes														*/
	IN 		BSI_GCT			ucTag,					/* Tag of data item to store														*/
 	IN 		BSI_BYTE_PTR	uszValue,				/* Data value to store																*/
	IN 		BSI_ULONG		unValueLen				/* Data value length in bytes														*/
); 
 

/* gscBsiGcDataDelete() 
 *
 * Purpose: Delete the data item associated with the tag value in the specified container. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_BAD_TAG 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_IO_ERROR 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGcDataDelete( 
	IN 		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect()									*/
	IN 		BSI_BYTE_PTR	uszAID, 				/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN 		BSI_ULONG		unAIDLen, 				/* AID value length in bytes														*/
	IN 		BSI_GCT			ucTag 					/* Tag of data item to delete														*/
); 


/*
 * gscBsiGcGetContainerProperties() 
 *
 * Purpose: Retrieves the properties of the specified container.  Access Control
 *		Rules are common to all data items managed by the selected container. 
 *
 * Returns: BSI_OK 
 *			BSI_BAD_HANDLE 
 *			BSI_SC_LOCKED
 *			BSI_BAD_AID 
 *			BSI_BAD_PARAM 
 *			BSI_CARD_REMOVED 
 *			BSI_NO_CARDSERVICE 
 *			BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGcGetContainerProperties( 
	IN		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect()			*/
	IN		BSI_BYTE_PTR	uszAID, 				/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN		BSI_ULONG		unAIDLen, 				/* AID value length in bytes								*/
	OUT		BSI_GC_ACR_PTR	strctGCacr, 			/* Structure indicating access control conditions for all 
														operations.												*/
	OUT		BSI_GC_SIZE_PTR	strctContainerSizes,	/* For Virtual Machine cards, the size(in bytes) of the 
														container specified by uszAID.  unMaxNbDataItems is the 
														size of the T-Buffer, and unMaxValueStorageSize is the 
														size of the V-Buffer.  For file system cards that cannot 
														calculate these values, both fields of this structure 
														will be set to 0.										*/
	OUT		BSI_BYTE_PTR	containerVersion 		/* Version of the container.  The format of this value is 
														application dependent.  In cases where the card cannot 
														return a container version, this string will contain 
														only the NULL terminator '\0'							*/
);

/*
 * gscBsiGcReadTagList() 
 *
 * Purpose: Return the list of tags in the selected container. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGcReadTagList( 
	IN BSI_CARD_HANDLE		hCard,					/* Card connection handle from gscBsiUtilConnect().									*/
	IN BSI_BYTE_PTR			uszAID,					/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN BSI_ULONG			unAIDLen,				/* AID value length in bytes														*/
	IN OUT BSI_GCT_PTR		TagArray,				/* An array containing the list of tags for the selected container					*/
	IN OUT BSI_ULONG_PTR	punNbTags 				/* Number of tags in TagArray														*/
); 


/*
 * gscBsiGcReadValue() 
 *
 * Purpose: Returns the Value associated with the specified Tag. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_BAD_TAG 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_IO_ERROR 
 *          BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscBsiGcReadValue( 
	IN BSI_CARD_HANDLE		hCard,				/* Card connection handle from gscBsiUtilConnect().									*/
	IN BSI_BYTE_PTR			uszAID, 			/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN BSI_ULONG			unAIDLen, 			/* AID value length in bytes														*/
	IN BSI_GCT				ucTag, 				/* Tag value of data item to read													*/
	IN OUT BSI_BYTE_PTR		uszValue, 			/* Value associated with the specified tag.  The caller must
													allocate the buffer.															*/
	IN OUT BSI_ULONG_PTR	punValueLen			/* Size of the buffer allocated by the caller to hold the 
													returned Value (input).  Size of the Value returned 
													(output).																		*/
); 
 

/*
 * gscBsiGcUpdateValue() 
 *
 * Purpose: Updates the Value associated with the specified Tag. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_BAD_TAG 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_NO_MORE_SPACE 
 *          BSI_IO_ERROR 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGcUpdateValue( 
	IN 		BSI_CARD_HANDLE	hCard,					/* Card connection handle from gscBsiUtilConnect().									*/
	IN 		BSI_BYTE_PTR	uszAID, 				/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN 		BSI_ULONG		unAIDLen, 				/* AID value length in bytes.														*/
	IN 		BSI_GCT			ucTag, 					/* Tag of data item to update.														*/
	IN 		BSI_BYTE_PTR	uszValue, 				/* New Value of the data item.														*/
	IN 		BSI_ULONG		unValueLen 				/* Length in bytes of the new Value.												*/
); 
 

/*
 * gscBsiGetChallenge() 
 *
 * Purpose: Retrieves a randomly generated challenge from the card as the 
 *			first step of a challenge-response authentication protocol between
 *			the client application and the card.  The client subsequently 
 *			encrypts the challenge using a symmetric key and returns the 
 *			encrypted random challenge to the card through a call to 
 *			gscBsiUtilAcquireContext() in the uszAuthValue field of a 
 *			BSIAuthenticator structure.   
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGetChallenge( 
	IN 		BSI_CARD_HANDLE		hCard,				/* Card connection handle from gscBsiUtilConnect()									*/
	IN 		BSI_BYTE_PTR		uszAID, 			/* Target  container AID value. The parameter shall be in ASCII hexadecimal format	*/
	IN 		BSI_ULONG			unAIDLen, 			/* AID value length in bytes														*/
	IN OUT 	BSI_BYTE_PTR		uszChallenge, 		/* Random challenge returned from the card											*/
	IN OUT 	BSI_ULONG_PTR		punChallengeLen		/* Length of random challenge in bytes												*/
); 
 

/*
 * gscBsiSkiInternalAuthenticate() 
 *
 * Purpose: Computes a symmetric key cryptogram in response to a challenge. 
 *			In cases where the card reader authenticates the card, this 
 *			function does not return a cryptogram.  In these cases a 
 *			BSI_TERMINAL_AUTH will be returned if the card reader successfully 
 * 			authenticates the card.  BSI_ACCESS_DENIED is returned if the card 
 *			reader fails to authenticate the card. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_BAD_AID
 *			BSI_SC_LOCKED
 *          BSI_BAD_PARAM 
 *          BSI_BAD_ALGO_ID 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_TERMINAL_AUTH 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscBsiSkiInternalAuthenticate( 
	IN BSI_CARD_HANDLE		hCard,					/* Card connection handle from gscBsiUtilConnect().			*/
 	IN BSI_BYTE_PTR			uszAID, 				/* SKI provider module AID value. The parameter shall be in ASCII hexadecimal format */
	IN BSI_ULONG			unAIDLen, 				/* AID value length in bytes								*/
	IN BSI_BYTE				ucAlgoID,				/* Identifies the cryptographic algorithm that the card must
													use to encrypt the challenge.  All conformant 
													implementations shall, at a minimum, support DES3-ECB 
													(Algorithm Identifier 0x81) and DES3-CBC (Algorithm 
													Identifier 0x82).  Implementations may optionally 
													support other cryptographic algorithms.					*/
 	IN BSI_BYTE_PTR			uszChallenge, 			/* Challenge generated by the client application and 
													submitted to the card									*/
	IN BSI_ULONG			unChallengeLen, 		/* Length of the challenge in bytes.						*/
	OUT BSI_BYTE_PTR		uszCryptogram,			/* The cryptogram computed by the card.						*/
 	IN OUT BSI_ULONG_PTR	punCryptogramLen 		/* Length of the cryptogram computed by the card in bytes.  */
); 
 

/*
 * gscBsiPkiCompute() 
 *
 * Purpose: Performs a private key computation on the message digest using the 
 *			private key associated with the specified AID. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_BAD_ALGO_ID 
 *          BSI_CARD_REMOVED 
 *          BSI_ACCESS_DENIED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscBsiPkiCompute( 
	IN BSI_CARD_HANDLE		hCard,				/* Card connection handle from gscBsiUtilConnect().			*/
	IN BSI_BYTE_PTR			uszAID, 			/* PKI provider module AID value. The parameter shall be in ASCII hexadecimal format */
	IN BSI_ULONG			unAIDLen, 			/* AID value length in bytes								*/
	IN BSI_BYTE				ucAlgoID, 			/* Identifies the cryptographic algorithm that will be used 
													to generate the signature.  All conformant 
													implementations shall, at a minimum, support RSA_NO_PAD 
													(Algorithm Identifier 0xA3).  Implementations may 
													optionally support other algorithms.
													Added support ECC_NO_PAD (Algorithm identifier 0xA4)  //Dinesh*/
	IN BSI_BYTE_PTR			uszMessage,			/* The hash of the message to be signed						*/
	IN BSI_ULONG			unMessageLen, 		/* Length of hashed message to be signed, in bytes.			*/
	IN OUT BSI_BYTE_PTR		uszResult, 			/* Buffer containing the signature.							*/
	IN OUT BSI_ULONG_PTR	punResultLen 		/* Length of the signature buffer in bytes					*/
);


/*
 * gscBsiPkiGetCertificate() 
 *
 * Purpose: Reads the certificate from the card.   
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_ACCESS_DENIED 
 *          BSI_IO_ERROR 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscBsiPkiGetCertificate( 
	IN BSI_CARD_HANDLE		hCard,					/* Card connection handle from gscBsiUtilConnect().			*/
	IN BSI_BYTE_PTR			uszAID, 				/* PKI provider module AID value. The parameter shall be in ASCII hexadecimal format */
	IN BSI_ULONG			unAIDLen, 				/* AID value length in bytes.								*/
	OUT BSI_BYTE_PTR		uszCertificate, 		/* Buffer containing the certificate.						*/
	IN OUT BSI_ULONG_PTR	punCertificateLen 		/* Length of the certificate buffer in bytes.				*/
); 
 

/*
 * gscBsiGetCryptoProperties() 
 *
 * Purpose: Retrieves the Access Control Rules and private cryptographic 
 *			key length managed by the PKI provider module. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_AID 
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscBsiGetCryptoProperties( 
	IN BSI_CARD_HANDLE		hCard,					/* Card connection handle from gscBsiUtilConnect().			*/
	IN BSI_BYTE_PTR			uszAID, 				/* AID of the PKI provider.	The parameter shall be in ASCII hexadecimal format */
	IN BSI_ULONG			unAIDLen, 				/* Length of the AID of the PKI provider, in bytes.			*/
	OUT BSI_CRYPTO_ACR_PTR	strctCRYPTOacr,			/* Structure indicating access control conditions for all 
													operations.  The range of possible values for the 
													members of this structure are defined in Table 3.2 
													(Section 3.1), and the allowable ACRs for each 
													function in Table 3.4 (Section 3.2):					*/
	OUT BSI_ULONG_PTR		punKeyLen				/* Length of the private key managed by the PKI provider.	*/
); 
 

/*
 * gscXsiUtilGetCardATR() 
 *
 * Purpose: Retrieves the ATR of the card. 
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *			BSI_SC_LOCKED
 *          BSI_BAD_PARAM 
 *          BSI_CARD_REMOVED 
 *          BSI_NO_CARDSERVICE 
 *          BSI_UNKNOWN_ERROR 
 *			BSI_INSUFFICIENT_BUFFER 
 */
BSI_RETURN CALLBACK gscXsiUtilGetCardATR( 
	IN BSI_CARD_HANDLE		hCard,					/* Card connection handle from gscBsiUtilConnect().			*/
	INOUT BSI_BYTE_PTR		psuATR, 				/* Retrieved ATR */
	INOUT BSI_ULONG			*punATRLen 				/* Retrieved ATR length */
);

/*
 * gscXsiUtilGetCuid
 *
 * Purpose: Retrieves the Card Unique ID
 *
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_CARD_REMOVED 
 *          BSI_BAD_PARAM 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscXsiUtilGetCuid( 
	IN		BSI_CARD_HANDLE	hCard,				/* Card connection handle */
	INOUT	BSI_BYTE_PTR	pusSerialNumber,	/* Card Serial Number */
	INOUT	BSI_ULONG_PTR	punSerialNumberLen	/* Serial Number length */
);

/*
 * gscXsiUtilGetIdentifier
 *
 * Purpose: Retrieves the card identifier 
 *
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE 
 *          BSI_CARD_REMOVED 
 *          BSI_BAD_PARAM 
 *          BSI_INSUFFICIENT_BUFFER 
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscXsiUtilGetIdentifier( 
	IN		BSI_CARD_HANDLE	hCard,				/* Card connection handle	*/
	INOUT	BSI_BYTE_PTR	pusIdentifier,		/* Card Identifier			*/
	INOUT	BSI_ULONG_PTR	punIdentifierLen	/* Serial Identifier length */
);

/*
 * gscXsiNotifySynchronizationStart
 *
 * Purpose: Starts to notify synchronisation 
 *
 *
 * Returns: BSI_NOT_IMPLEMENTED
 */
BSI_RETURN CALLBACK gscXsiNotifySynchronizationStart( 
	IN		BSI_CARD_HANDLE	hCard				/* Card connection handle	*/
);

/*
 * gscXsiNotifySynchronizationEnd
 *
 * Purpose: Stops to notify synchronisation
 *
 *
 * Returns: BSI_NOT_IMPLEMENTED
 */
BSI_RETURN CALLBACK gscXsiNotifySynchronizationEnd( 
	IN		BSI_CARD_HANDLE	hCard				/* Card connection handle	*/
);

/*
 * gscXsiUtilGetCardManagerState
 *
 * Purpose: Retrieves the Card Manager state
 *
 *
 * Returns: BSI_OK 
 *          BSI_BAD_HANDLE
 *          BSI_CARD_REMOVED
 *          BSI_BAD_PARAM
 *          BSI_UNKNOWN_ERROR 
 */
BSI_RETURN CALLBACK gscXsiUtilGetCardManagerState( 
	IN		BSI_CARD_HANDLE	hCard,				/* Card connection handle */
	OUT		BSI_INT_PTR		piState				/* Retrieved state: BSI_CARD_OP_READY or BSI_CARD_LOCKED */
);

/*
 * gscXsiUtilChangePIN() 
 *
 * Purpose: Change the PIN of the card. 
 *
 * Returns:	BSI_OK
 *			BSI_BAD_HANDLE 
 *			BSI_BAD_AID 
 *			BSI_ACR_NOT_AVAILABLE 
 *			BSI_BAD_AUTH 
 *			BSI_CARD_REMOVED 
 *			BSI_PIN_BLOCKED 
 *			BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscXsiUtilChangePIN( 
	IN BSI_CARD_HANDLE	hCard,			// card connection handle from gscBsiUtilConnect()
	IN BSI_BYTE_PTR		uszAID, 		// AID of PIN/ACA applet instance
	IN BSI_ULONG		unAIDLen, 		// length of AID of PIN/ACA applet instance
	IN BSI_BYTE_PTR		uszOldPIN, 		// current PIN
	IN BSI_ULONG		unOldPINLen, 	// length of current PIN
	IN BSI_BYTE_PTR		uszNewPIN,		// new PIN
	IN BSI_ULONG		unNewPINLen 	// length of new PIN
);

/*
 * gscXsiUtilIsPINVerified() 
 *
 * Purpose: Check if the PIN has been presented to the card 
 *
 * Returns:	BSI_OK
 *			BSI_BAD_HANDLE 
 *			BSI_BAD_AID 
 *			BSI_ACR_NOT_AVAILABLE 
 *			BSI_BAD_AUTH 
 *			BSI_CARD_REMOVED 
 *			BSI_PIN_BLOCKED 
 *			BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscXsiUtilIsPINVerified( 
	IN	BSI_CARD_HANDLE	hCard,			// card connection handle from gscBsiUtilConnect()
	IN	BSI_BYTE_PTR	uszAID, 		// AID of PIN/ACA applet instance
	IN	BSI_ULONG		unAIDLen, 		// length of AID of PIN/ACA applet instance
	OUT	int*			pnLeftPINTries	// number of left PIN tries	
);

/*
 * gscXsiUtilGetForceChangePIN() 
 *
 * Purpose: Retrieve the Force Change PIN flag 
 *
 * Returns:	BSI_OK
 *			BSI_BAD_HANDLE 
 *			BSI_BAD_AID 
 *			BSI_ACR_NOT_AVAILABLE 
 *			BSI_CARD_REMOVED 
 *			BSI_UNKNOWN_ERROR
 */
BSI_RETURN CALLBACK gscXsiUtilGetForceChangePIN( 
	IN	BSI_CARD_HANDLE	hCard,				// card connection handle from gscBsiUtilConnect()
	IN	BSI_BYTE_PTR	uszAID, 			// AID of PIN/ACA applet instance
	IN	BSI_ULONG		unAIDLen, 			// length of AID of PIN/ACA applet instance
	OUT	BOOL*			pbForceChangePIN	// TRUE if the Force Change PIN flag is activated for this AID
);

#ifdef __cplusplus
}
#endif

#endif /*__BSIF_H__*/

