/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */
 
#include "stdafx.h"

#include "Helper.h"
#include "Scenarios.UI.h"
#include "Scenarios.h"


extern BSI_CARD_HANDLE	g_hCard;

/***************/
/** FUNCTIONS **/
/***************/

/**
* EstablishCommunicationWithCard
*
* Establish a communication with a card
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN EstablishCommunicationWithCard()
{
	BSI_RETURN		lResp = BSI_OK;
	TCHAR			szInputBuffer[BUFSIZ];	
	unsigned int 	unReadersCount=0;
	BSI_BYTE_PTR	usReaderList=NULL; 
	BSI_INT			unReaderListLen=0; 
	ReaderStruct	sReaderStruct[10];	
	unsigned int	unIndex = 0;

	if (g_hCard != -1)
	{
		_tprintf(_T("\n Already connected, first disconnect\n"));
		lResp = gscBsiUtilDisconnect(g_hCard);
		DisplayError(" ... ", lResp);
		if (lResp != BSI_OK)
		{
			_tprintf(_T("\nCalling gscBsiUtilDisconnect failed."));
			PrintExtendedErrorText();
			goto end_function;
		}
		g_hCard = (BSI_CARD_HANDLE) -1;		
	}


	unReadersCount=0;
	/* use 	gscBsiUtilGetReadersList to retrieve the list of configured readers */		
	/* first get the length of readers list : provide NULL buffer for readers list */
	lResp = gscBsiUtilGetReaderList(
				NULL,	
				&unReaderListLen	
				);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiUtilGetReaderList failed."));
		PrintExtendedErrorText();
		goto end_function;
	}	
	else	
	{
		if (unReaderListLen > 2) /* two trailing '\0'  at least */
		{	
			/* allocate buffer to actually read readers list */
			usReaderList = (unsigned char *)malloc(unReaderListLen*sizeof(unsigned char));
			if (usReaderList != NULL)
			{
				/* now retrieve readers list */
				lResp = gscBsiUtilGetReaderList(
							usReaderList,
							&unReaderListLen
							);
				if (lResp != BSI_OK)
				{
					_tprintf(_T("\nCalling gscBsiUtilGetReaderList failed"));
					PrintExtendedErrorText();
					goto end_function;
				}
				else
				{
					_tprintf(_T("\nSelect a reader:"));
					DisplayReadersList(usReaderList, unReaderListLen, sReaderStruct, &unReadersCount);
				}
				free(usReaderList);
				usReaderList = NULL;
			}	
		}
		else
		{
			_tprintf(_T("\nNo available readers...."));
		}

	}

	/* now get status */
	if (unReadersCount !=0)
	{
		/* select one among available reads */
		GetString(_T("\nEnter Reader choice"), szInputBuffer, BUFSIZ);
		_stscanf_s(szInputBuffer, _T("%d"), &unIndex);
		if (unIndex <= 0 || unIndex > unReadersCount)
		{
			_tprintf(_T("\nInvalid Choice... Exiting scenario."));
			lResp = BSI_UNKNOWN_ERROR;
			goto end_function;
		}
		else
		{
			/* perform status for reader  : connection handle is set to invalid value */
			lResp = gscBsiUtilConnect(
						sReaderStruct[unIndex-1].usReader,
						sReaderStruct[unIndex-1].unReaderLen+1,
						&g_hCard);
			if (lResp != BSI_OK)
			{
				_tprintf(_T("\nCalling gscBsiUtilConnect failed."));
				PrintExtendedErrorText();
				goto end_function;
			}
		}
	}

	_tprintf(_T(" Card Connection handle : %d\n"), g_hCard);

end_function:
	return lResp;
}

/**
* CloseCommunicationCard
*
* Close communication to the card
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
void CloseCommunicationCard()
{
	BSI_RETURN	lResp = BSI_OK;

	if (g_hCard == -1)
	{
		_tprintf(_T("\n\tNo current connection "));
	}
	else
	{

		lResp = gscBsiUtilDisconnect(g_hCard);
		if (lResp != BSI_OK)
		{
			_tprintf(_T("\nCalling gscBsiUtilDisconnect failed."));
			PrintExtendedErrorText();
		}
		
		g_hCard = (BSI_CARD_HANDLE) -1;		
	}
}


/**
* GetAccessToServicesUsingPIN
*
* Use PIN authenticator to grant access to services
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN GetAccessToServicesUsingPIN(BSI_BYTE_PTR usAID, BSI_INT unAIDLen)
{
	BSI_RETURN		lResp = BSI_OK;
	TCHAR			szInputBuffer[BUFSIZ];
	char			szPIN[BUFSIZ];
	size_t			pinLen = 0;
	char			szPinHex[BUFSIZ];
	int				nPinHexSize = 0;
	BSI_AUTH_PTR	pstrAuthent=NULL;
	BSI_USHORT		unAuthenticatorNb=1;

	pstrAuthent = (BSIAuthenticator *)malloc(sizeof(BSIAuthenticator));
	if (pstrAuthent == NULL)
	{
		_tprintf(_T("\nOut of memory..."));
		goto end_function;
	}
	else
	{
		GetString(_T("\nEnter PIN (alphanumeric string; ex.: \"1234\"): "), szInputBuffer, BUFSIZ);
		/* convert to hexa string */
		memset(szPinHex, 0, BUFSIZ*sizeof(char));
		nPinHexSize = BUFSIZ*sizeof(char);
		wcstombs_s (&pinLen, szPIN, sizeof (szPIN), szInputBuffer, BUFSIZ);
		ConvBinToAscii(szPIN, (int)strnlen_s(szPIN, sizeof(szPIN)), szPinHex, &nPinHexSize);
		strcpy_s((char *) pstrAuthent->uszAuthValue, sizeof(pstrAuthent->uszAuthValue), szPinHex);
			
		pstrAuthent->unAuthValueLen = (ULONG) strlen(szPinHex) + 1;

		pstrAuthent->unAccessMethodType=BSI_AM_PIN;
		pstrAuthent->unKeyIDOrReference=1;
	}

	lResp = gscBsiUtilAcquireContext(
				g_hCard,
				usAID,
				unAIDLen,
				pstrAuthent,
				unAuthenticatorNb);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiUtilAcquireContext failed."));
		PrintExtendedErrorText();
		goto end_function;
	}
	

end_function:
	if (pstrAuthent != NULL)
	{
		free(pstrAuthent);
		pstrAuthent = NULL;
	}

	return lResp;
}


/**
* GiveUpGrantedAccessToContainerServices
*
* Give up granted access to container service
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN GiveUpGrantedAccessToContainerServices(BSI_BYTE_PTR usAID, BSI_INT unAIDLen)
{
	BSI_RETURN		lResp = BSI_OK;

	lResp = gscBsiUtilReleaseContext(
				g_hCard,
				usAID,
				unAIDLen);
	DisplayError(" ... ", lResp);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiUtilReleaseContext failed."));
		PrintExtendedErrorText();
	}
	
	return lResp;
}	



/**
* GetAPIVersion
*
* Get API version 
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN GetAPIVersion()
{
	BSI_RETURN		lResp = BSI_OK;
	BSI_BYTE		usVersion[BUFSIZ];
	BSI_INT			nVersionLen = BUFSIZ;

	
	/* initialize input data length */
	nVersionLen = sizeof(usVersion);
	/* read data */
	lResp = gscBsiUtilGetVersion(
				usVersion,
				&nVersionLen);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiUtilGetVersion"));
		PrintExtendedErrorText();
	}
	else	
	{
		_tprintf(_T("\nBSI Provider API v%S"), usVersion);
	}

	return lResp;
}


BSI_RETURN GetGCContainerACR(BSI_BYTE_PTR usAID, BSI_INT unAIDLen, BSI_LONG_PTR punACRType)
{
	BSI_RETURN		lResp = BSI_OK;
	BSI_GC_ACR		strACR;

	TCHAR szCreateACR[BUFSIZ];
	TCHAR szDeleteACR[BUFSIZ];
	TCHAR szReadTagListACR[BUFSIZ];
	TCHAR szReadValueACR[BUFSIZ];
	TCHAR szUpdateValueACR[BUFSIZ];
	
	/* read Acces Right only */ 
	lResp = gscBsiGcGetContainerProperties(
				g_hCard,
				usAID,
				unAIDLen,
				&strACR,
				NULL,
				NULL);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiGcGetContainerProperties failed"));
		PrintExtendedErrorText();
		goto end_function;
	}
	else
	{

		AccessControlRuleToString(strACR.strctCreateACR.unACRType, szCreateACR);
		AccessControlRuleToString(strACR.strctDeleteACR.unACRType, szDeleteACR);
		AccessControlRuleToString(strACR.strctReadTagListACR.unACRType, szReadTagListACR);
		AccessControlRuleToString(strACR.strctReadValueACR.unACRType, szReadValueACR);
		AccessControlRuleToString(strACR.strctUpdateValueACR.unACRType, szUpdateValueACR);

		_tprintf(_T("\nRead tag list ACR\t: %s"), szReadTagListACR);
		_tprintf(_T("\nRead value ACR\t\t: %s"), szReadValueACR);
		_tprintf(_T("\nUpdate value ACR\t: %s"), szUpdateValueACR);

		if ( (strACR.strctReadTagListACR.unACRType == BSI_ACR_ALWAYS && strACR.strctReadValueACR.unACRType == BSI_ACR_PIN)
			|| (strACR.strctReadTagListACR.unACRType == BSI_ACR_PIN && strACR.strctReadValueACR.unACRType == BSI_ACR_ALWAYS)
			|| (strACR.strctReadTagListACR.unACRType == BSI_ACR_PIN && strACR.strctReadValueACR.unACRType == BSI_ACR_PIN) )
		{
			*punACRType = BSI_ACR_PIN;
		}
		else if (strACR.strctReadValueACR.unACRType == BSI_ACR_ALWAYS && strACR.strctReadValueACR.unACRType == BSI_ACR_ALWAYS)
		{
			*punACRType = BSI_ACR_ALWAYS;
		}
		else
		{
			lResp = BSI_UNKNOWN_ERROR;
			goto end_function;
		}

	}

end_function:
	return lResp;
}


/**
* ListOfTagsInContainer
*
* List available tags in container
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN ListOfTagsInContainer(BSI_BYTE_PTR usAID, BSI_INT unAIDLen, BSI_GCT_PTR* ppTagArray, BSI_UINT* punTagNb)
{
	BSI_RETURN		lResp = BSI_OK;
	//int	i;

	lResp = gscBsiGcReadTagList(
				g_hCard,
				usAID,
				unAIDLen,
				NULL,
				punTagNb
				);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiGcReadTagList failed."));
		PrintExtendedErrorText();
		goto end_function;
	}
	else	
	{
		_tprintf(_T("\n\t %d Tag(s) in container"), *punTagNb); 
		if (*punTagNb != 0)
		{	
			*ppTagArray = (BSI_GCT*)malloc((*punTagNb) * sizeof(BSI_GCT));
			if (*ppTagArray != NULL)
			{
				
				lResp = gscBsiGcReadTagList(
							g_hCard,
							usAID,
							unAIDLen,
							*ppTagArray,
							punTagNb);
				if (lResp != BSI_OK)
				{
					_tprintf(_T("\nCalling gscBsiGcReadTagList failed"));
					PrintExtendedErrorText();
					goto end_function;
				}	
				else	
				{
					/*for (i = 0 ; (unsigned int) i < *punTagNb ; i++)
					{
						printf("\n\tTag #%d: 0x%02x", i+1, (*ppTagArray)[i]);
					}*/
					;
				}
			}
		}
	}

end_function:
	return lResp;
}

/**
* ReadDataInContainer
*
* Read data in container
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN ReadDataInContainer(BSI_BYTE_PTR usAID, BSI_INT unAIDLen, BSI_GCT ucTag)
{
	BSI_RETURN		lResp = BSI_OK;
	BSI_BYTE_PTR	usDataReadValue=NULL; 
	BSI_UINT		unDataReadValueLen=0; 

	/* retrieve data length */
	lResp = gscBsiGcReadValue(
				g_hCard,
				usAID,
				unAIDLen,
				ucTag,
				NULL,
				&unDataReadValueLen
				);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiGcReadValue failed."));
		PrintExtendedErrorText();
		goto end_function;
	}	
	else	
	{
		_tprintf(_T("\t Data value length: %d\n"), unDataReadValueLen); 
		if (unDataReadValueLen != 0)
		{	
			usDataReadValue = (unsigned char *)malloc(unDataReadValueLen * sizeof(unsigned char));
			if (usDataReadValue != NULL)
			{
				/* retrieve data */
				lResp = gscBsiGcReadValue(
							g_hCard,
							usAID,
							unAIDLen,
							ucTag,
							usDataReadValue,
							&unDataReadValueLen
							);
				if (lResp != BSI_OK)
				{
					printf("Calling gscBsiGcReadValue failed.");
					PrintExtendedErrorText();
					goto end_function;
				}	
				else	
				{
					_tprintf(_T("\tData Value: %*.*S|\n"), unDataReadValueLen, unDataReadValueLen, usDataReadValue );
				}
				free(usDataReadValue);
				usDataReadValue = NULL;
			}
		}
	}

end_function:
	return lResp;
}



BSI_RETURN GetPKIContainerACR(BSI_BYTE_PTR usAID, BSI_INT unAIDLen, BSI_ULONG_PTR punACRType, BSI_ULONG_PTR punKeyLen)
{
	BSI_RETURN		lResp = BSI_OK;
	BSI_CRYPTO_ACR	strCryptoACR;
	TCHAR			szPKIComputeACR[BUFSIZ];

	_tprintf(_T("\nGetting ACR for applet\t: %S"), usAID);

	/* read Acces Right only */
	lResp = gscBsiGetCryptoProperties(
				g_hCard,
				usAID,
				unAIDLen,	
				&strCryptoACR,
				punKeyLen);
	if (lResp != BSI_OK)
	{
		printf("\nCalling gscBsiGetCryptoProperties failed.");
		PrintExtendedErrorText();
		goto end_function;
	}
	else
	{
		*punACRType = strCryptoACR.strctPkiComputeACR.unACRType;
		AccessControlRuleToString(strCryptoACR.strctPkiComputeACR.unACRType, szPKIComputeACR);
		_tprintf(_T("\nCompute PKI service ACR\t: %s\n"), szPKIComputeACR);
		_tprintf(_T("\nKey Length: \t: %d\n"), *punKeyLen);
	}

end_function:
	return lResp;
}


/**
* BsiPkiCompute
*
* Computes the private key 
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
BSI_RETURN BsiPkiCompute(BSI_BYTE_PTR usAID, BSI_INT unAIDLen, BSI_ULONG unKeyLen)
{
	BSI_RETURN		lResp = BSI_OK;
	/* buffer to be signed 
	BSI_BYTE		usDataToBeSigned[]= { \
		0x00, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, \
		0x81, 0x8c, 0x6e, 0x60, 0x1a, 0x24, 0xf7, 0x27, 0x50, 0xda, 0x0f, 0x6c, 0x9b, 0x8e, 0xbe, 0x28};
	** converted to the following string :
	*/
	int i = 0;
	BSI_BYTE_PTR	usDataToBeSigned	= NULL;
	BSI_BYTE_PTR	usSignedData		= NULL;
	BSI_INT			unSignedDataLen		= 0;
	BSI_BYTE_PTR	usDataHexAscii		= NULL;
	BSI_INT			unDataHexAsciiLen	= 2*unKeyLen;
	// md5 for message "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua."
	// "818c6e601a24f72750da0f6c9b8ebe28";
	BSI_BYTE digest[] = {0x81, 0x8c, 0x6e, 0x60, 0x1a, 0x24, 0xf7, 0x27, 0x50, 0xda, 0x0f, 0x6c, 0x9b, 0x8e, 0xbe, 0x28};

	// pad according to RFC
	//int msg_len = (int) strlen(digest);
	int msg_len = (int) sizeof(digest);
	int pad_len = unKeyLen - 2 - msg_len;
	usDataToBeSigned = (BSI_BYTE_PTR) malloc ((unKeyLen)*sizeof(BSI_BYTE));
	usSignedData     = (BSI_BYTE_PTR) malloc ((2*unKeyLen+1)*sizeof(BSI_BYTE));
	usDataHexAscii   = (BSI_BYTE_PTR) malloc ((2*unKeyLen+1)*sizeof(BSI_BYTE));
	if (NULL==usDataToBeSigned || NULL==usSignedData || NULL==usDataHexAscii)
	{
		_tprintf(_T("Failed to allocate memory.\n"));
		lResp = BSI_UNKNOWN_ERROR;
		goto end_function;
	}

	memset (usDataToBeSigned, 0xFF, unKeyLen);
	usDataToBeSigned[0] = 0x00;
	usDataToBeSigned[1] = 0x01;
	for ( i=0; i < sizeof (digest); i++)
	{
		((BSI_BYTE_PTR)usDataToBeSigned+pad_len+2)[i] = digest[i];
	}
	
	unSignedDataLen = 2*unKeyLen+1;
	memset (usDataHexAscii, 0, 2*unKeyLen+1);
	if (ConvBinToAscii ( (char*)usDataToBeSigned, unKeyLen, (char*)usDataHexAscii, (int*)&unDataHexAsciiLen)==-1)
	{
		_tprintf(_T("Error converting message from binary to hexadecimal\n"));
		lResp = BSI_UNKNOWN_ERROR;
		goto end_function;
	}

	unDataHexAsciiLen = 2*unKeyLen+1;
	_tprintf(_T("Unsigned data (hexadecimal): %S\n"), usDataHexAscii);

	lResp = gscBsiPkiCompute(
				g_hCard,
				usAID,
				unAIDLen,
				BSI_RSA_NO_PAD,
				usDataHexAscii,
				unDataHexAsciiLen,
				usSignedData,
				&unSignedDataLen
				);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiPkiCompute failed."));
		PrintExtendedErrorText();
		goto end_function;
	}	
	else
	{
		_tprintf(_T("Signed data len: %d (i.e. %d bytes)\n"), unSignedDataLen, ((unSignedDataLen-1)/2));
		_tprintf(_T("Signed data (hexadecimal): %S"), usSignedData);
	}

end_function:
	if (usDataToBeSigned!=NULL) {
		free (usDataToBeSigned);
		usDataToBeSigned=NULL;
	}
	if (usDataHexAscii!=NULL) {
		free (usDataHexAscii);
		usDataHexAscii=NULL;
	}
	if (usSignedData!=NULL) {
		free (usSignedData);
		usSignedData=NULL;
	}
	return lResp;
}


/**
* PrintExtendedErrorText
*
* PrintExtendedErrorText
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
void PrintExtendedErrorText()
{
	BSI_RETURN	lResp = BSI_OK;
	char		szErrorText[BSI_ERROR_TEXT_LEN];	

	lResp = gscBsiUtilGetExtendedErrorText(
				g_hCard,
				szErrorText
				);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("\nCalling gscBsiUtilGetExtendedErrorText failed."));
		_tprintf(_T("\nFailed to get last error description\n"));
	}	
	else
	{
		_tprintf(_T("\nLast BSI return code diagnosis: |%S|"), szErrorText);
	}
}


/**
* ScenarioSignData
*
* ScenarioSignData
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
void ScenarioSignData()
{
	BSI_BYTE		usAID[BUFSIZ];
	BSI_INT			unAIDLen=0;
	BSI_BYTE		usPINAID[BUFSIZ];
	BSI_INT			unPINAIDLen=0;
	BSI_ULONG		unACRType=0;
	BSI_ULONG		unKeyLen = 0;

	_tprintf(_T("\nStarting Sign Data Scenario\n"));
	// connect
	if (EstablishCommunicationWithCard() != BSI_OK || g_hCard==-1){
		goto end_function;
	}
	
	// get PKI container AID
	/* AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1 */
	unAIDLen = sizeof(usAID);
	GetContainerAID(_T("\nEnter PKI Container AID (hexadecimal string; ex.: \"A0000000790300\")"), usAID, &unAIDLen);
	
	// get ACR for applet
	if (GetPKIContainerACR(usAID, unAIDLen, &unACRType, &unKeyLen) != BSI_OK) {
		goto exit_after_connection;
	}

	if (unACRType!=BSI_ACR_ALWAYS && unACRType!=BSI_ACR_PIN)
	{
		_tprintf (_T("The Sample only supports BSI_ACR_ALWAYS or BSI_ACR_PIN ACRs\n"));
		goto exit_after_connection;
	}

	// acquire context
	if (unACRType==BSI_ACR_PIN)
	{
		// get PIN applet AID
		/* AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1 */
		unPINAIDLen = sizeof(usPINAID);
		GetContainerAID(_T("\nEnter PIN Container AID (hexadecimal string; ex.: \"A0000000790300\")"), usPINAID, &unPINAIDLen);

		// get access using PIN
		if (GetAccessToServicesUsingPIN(usPINAID, unPINAIDLen) != BSI_OK) {
			goto exit_after_connection;
		}
	}

	// sign content
	if (BsiPkiCompute(usAID, unAIDLen, unKeyLen) != BSI_OK) {
		goto exit_after_auth;
	}

exit_after_auth:
	// release context
	if (unACRType==BSI_ACR_PIN)
	{
		GiveUpGrantedAccessToContainerServices(usPINAID, unPINAIDLen);
	}

exit_after_connection:
	// disconnect
	CloseCommunicationCard();

end_function:
	_tprintf(_T("\n\nFinished Sign Data Scenario\n"));
}


/**
* ScenarioReadData
*
* ScenarioReadData
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
void ScenarioReadData()
{
	BSI_GCT_PTR pTagArray = NULL;
	BSI_UINT unTagNb = 0;
	BSI_BYTE		usAID[BUFSIZ];
	BSI_INT			unAIDLen=0;
	BSI_BYTE		usPINAID[BUFSIZ];
	BSI_INT			unPINAIDLen=0;
	BSI_LONG		unACRType=0;
	unsigned int i = 0;

	_tprintf(_T("\nStarting Read Data Scenario\n"));
	// connect
	if (EstablishCommunicationWithCard() != BSI_OK || g_hCard==-1){
		goto end_function;
	}
	
	// get the GC Container AID
	/* AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1 */
	unAIDLen = sizeof(usAID);
	GetContainerAID(_T("\nEnter Generic Container AID (hexadecimal string; ex.: \"A0000000790300\")"), usAID, &unAIDLen);
	
	if (GetGCContainerACR(usAID, unAIDLen, &unACRType)!=BSI_OK)
	{
		goto exit_after_connection;
	}

	if (unACRType!=BSI_ACR_ALWAYS && unACRType!=BSI_ACR_PIN)
	{
		_tprintf (_T("The Sample only supports BSI_ACR_ALWAYS or BSI_ACR_PIN ACRs\n"));
		goto exit_after_connection;
	}

	if (unACRType==BSI_ACR_PIN)
	{
		// get PIN applet AID
		/* AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1 */
		unPINAIDLen = sizeof(usPINAID);
		GetContainerAID(_T("\nEnter PIN Container AID (hexadecimal string; ex.: \"A0000000790300\")"), usPINAID, &unPINAIDLen);

		// acquire context
		if (GetAccessToServicesUsingPIN(usAID, unAIDLen) != BSI_OK) {
			goto exit_after_connection;
		}
	}

	// read tag list
	if (ListOfTagsInContainer(usAID, unAIDLen, &pTagArray, &unTagNb) != BSI_OK) {
		if(pTagArray != NULL) { free(pTagArray); pTagArray = NULL;}
		goto exit_after_connection;
	}

	// read tag values
	for (i=0; i<unTagNb; i++){
		_tprintf(_T("%d, reading data for tag %x:\n"), i+1, pTagArray[i]);
		if (ReadDataInContainer(usAID, unAIDLen, pTagArray[i]) != BSI_OK) {
			goto exit_after_auth;
		}
	}

exit_after_auth:
	// release context
	if (unACRType==BSI_ACR_PIN)
	{
		GiveUpGrantedAccessToContainerServices(usAID, unAIDLen);
	}

exit_after_connection:
	// disconnect
	CloseCommunicationCard();

end_function:
	// cleanup 
	if(pTagArray != NULL) { free(pTagArray); pTagArray = NULL;}
	_tprintf(_T("\n\nFinished Read Data Scenario\n"));
}


/**
* ScenarioPassthru
*
* ScenarioPassthru
*
*@return ...
*
*@param param1 ...
*@param param2 ...
*
*@see ...
*
*@version 1.0
**/
void ScenarioPassthru()
{
	BSI_BYTE		szInputBuffer[BUFSIZ];
	BSI_BYTE		szResponse[BUFSIZ];
	BSI_INT			nResponseLen;
	BSI_INT			nCopyMaxLen;
	BSI_BYTE		usAID[BUFSIZ];
	BSI_INT			unAIDLen=0;
	BSI_RETURN		lResp = BSI_OK;
	BOOL			bIsBlocking = TRUE; // Non-blocking transactions not supported

	// init
	_tprintf(_T("\nStarting Send APDU Scenario\n"));
	// connect
	if (EstablishCommunicationWithCard() != BSI_OK || g_hCard==-1)
	{
		goto end_function;
	}

	// start transaction
	lResp = gscBsiUtilBeginTransaction(g_hCard, bIsBlocking);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("Calling gscBsiUtilBeginTransaction failed."));
		PrintExtendedErrorText();
		goto exit_after_connection;
	}

	// notifysynchronisationstart
	lResp = gscXsiNotifySynchronizationStart(g_hCard);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("Calling gscXsiNotifySynchronizationStart failed."));
		PrintExtendedErrorText();
		goto exit_after_connection;
	}
	
	// get applet id
	/* AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1 */
	unAIDLen = sizeof(usAID);
	GetContainerAID(_T("\nEnter PIN Container AID (hexadecimal string; ex.: \"A0000000790300\")"), usAID, &unAIDLen);

	/*printf("\n\tExample1: the APDU 0x00 0xA4 0x04 0x00 0x07 0xA0 0x00 0x00 0x00 0x79 0x03 0x00");
	printf("\n\tselects the applet whose ID is A0000000790300.");
	printf("\n\tThis APDU should be entered as 00A4040007A0000000790300");*/

	// send select applet passthru
	nResponseLen = BUFSIZ;
	strcpy_s((char*)szInputBuffer, BUFSIZ, "00A4040007");
	nCopyMaxLen = (BSI_INT)(BUFSIZ - strnlen_s((char*)szInputBuffer, BUFSIZ));
	strcpy_s((char*)szInputBuffer+strnlen_s((char*)szInputBuffer, BUFSIZ), nCopyMaxLen, (char*)usAID);
	_tprintf(_T("Sending Select Applet APDU: %S\n"), szInputBuffer);

	lResp = gscBsiUtilPassthru(
				g_hCard,
				szInputBuffer,
				(ULONG) strlen((char *) szInputBuffer)+1,
				szResponse,
				&nResponseLen);

	if (lResp != BSI_OK)
	{
		_tprintf(_T("Calling gscBsiUtilPassthru failed."));
		PrintExtendedErrorText();
		goto exit_after_connection;
	}
	else	
	{
		_tprintf(_T("\tResponse len: "));
		_tprintf(_T("%d (i.e. %d bytes)\n"), nResponseLen, ((nResponseLen-1)/2));
		_tprintf(_T("\tResponse : "));
		szResponse[nResponseLen] = 0;
		_tprintf(_T("%S\n"), szResponse);
		
		// expecting "61XX" or "...9000" for T0 or T1 type communication respectively
		if ( nResponseLen < 5 || (strncmp ((char*)szResponse+(nResponseLen-5), "90", 2) && strncmp ((char*)szResponse+(nResponseLen-5), "61", 2)) )
		{
			_tprintf(_T("\tError: Invalid Response\n"));
			goto exit_after_connection;
		}
	}


	// send ispinverified passthru
	nResponseLen = BUFSIZ;
	strcpy_s((char*)szInputBuffer, BUFSIZ, "8020000000");
	
	_tprintf(_T("\nSending IsPinVerified APDU: %S\n"), szInputBuffer);
	lResp = gscBsiUtilPassthru(
				g_hCard,
				szInputBuffer,
				(ULONG) strlen((char *) szInputBuffer)+1,
				szResponse,
				&nResponseLen);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("Calling gscBsiUtilPassthru failed."));
		PrintExtendedErrorText();
		goto exit_after_connection;
	}
	else	
	{
		_tprintf(_T("\tResponse len: "));
		_tprintf(_T("%d (i.e. %d bytes)\n"), nResponseLen, ((nResponseLen-1)/2));
		_tprintf(_T("\tResponse : "));
		szResponse[nResponseLen] = 0;
		_tprintf(_T("%S\n"), szResponse);
	}

	// end transaction
	lResp = gscBsiUtilEndTransaction(g_hCard);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("Calling gscBsiUtilEndTransaction failed."));
		PrintExtendedErrorText();
		goto exit_after_connection;
	}

	// notifisychronisationend
	lResp = gscXsiNotifySynchronizationEnd(g_hCard);
	if (lResp != BSI_OK)
	{
		_tprintf(_T("Calling gscXsiNotifySynchronizationEnd failed."));
		PrintExtendedErrorText();
		goto exit_after_connection;
	}

exit_after_connection:
	// disconnect
	CloseCommunicationCard();


end_function:
	// end
	_tprintf(_T("\n\nFinished Send APDU Scenario\n"));
	return;
}

