/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */

package bsi;

import java.util.Arrays;
import java.util.Vector;

import ui.MainWindow;
import ui.ScenarioInterface;
import ui.MainWindow.LogType;
import gov.gsc.classes.BSIAccessMethodType;
import gov.gsc.classes.BSIAuthenticator;
import gov.gsc.classes.BSIException;
import gov.gsc.classes.ContainerProperties;
import gov.gsc.classes.CryptoProperties;
import gov.gsc.classes.GSCBSI;
import gov.gsc.interfaces.BSIAccessControlRules;
import gov.gsc.interfaces.CryptographicAlgoID;



public class Scenarios implements ScenarioInterface
{
	private int m_hCard = -1;
	
	// List that will contain the list of smart card readers on the machine.
	private String [] m_readerList = null;
	
	private MainWindow m_mainWindow;
	
	// The wrapper object
	public GSCBSI	m_bsi = null;
	
	public Scenarios()
	{
		m_mainWindow = new MainWindow(this, new String[]{"Data signature", "Data items read", "APDUs sending"}, "HID Global BSI API Java sample");
		
		// Instantiate the BSI API Wrapper class.
		try
		{

			m_bsi = new GSCBSI();
			
		    if (m_bsi == null)
		    {
		    	m_mainWindow.print("Failed to initialize BSI API", LogType.ScenarioError);
            }
		    else
		    {
		    	m_mainWindow.print("------------------------------------------------------------------------------------");
		    	m_mainWindow.print("HID Global BSI API Java Sample");
		    	m_mainWindow.print("Copyright  2016 HID Global Corporation/ASSA ABLOY AB. All rights reserved.");
		    	m_mainWindow.print("------------------------------------------------------------------------------------");
		    }
		}
        catch (Exception ex)
		{
        	m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.ScenarioError);
            System.exit(-1);
        }
		
		// get BSI version
		String apiVersion = null;
		try
		{
			apiVersion = m_bsi.gscBsiUtilGetVersion();
		}
        catch (Exception ex)
		{
        	m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.ScenarioError);
            System.exit(-1);
        }
		
		// display start popup
		String popupMsg = 		"This sample demonstrates how to use the BSI API version 2.1 (\""+apiVersion+"\") provided by ActivClient 7.1 from a Java program.\n"
							+	"It implements following scenarios:\n"
							+	"        -    How to use a private key to sign some data.\n"
							+	"        -    How to read data items from the card.\n"
							+	"        -    How to send APDUs to an applet instance.\n";
		m_mainWindow.showMessageDialog(popupMsg);

	}
	
	/**
    * Executes the steps of the BSI signature scenario.
    *
    */
	public void scenarioSignData()
	{
		String pkiAID = "";
		String pinAID = "";		
		reftype authTypeByRef  = new reftype();
		authTypeByRef.typeAuth = 0;
		authTypeByRef.keylen   = -1;
		boolean connected = false;
		
		m_mainWindow.print("Starting Data Sign scenario.", LogType.ScenarioBoundary);
		
		// connect
		m_mainWindow.print("Try connecting to reader ...", LogType.ScenarioInfo);
		if(preamble()!=true)
		{
			m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		// AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1
		pkiAID = m_mainWindow.showAIDDialog("Enter PKI Applet ID:");
		m_mainWindow.print("pki AID: " + pkiAID);
		if (pkiAID == null)
        {
        	m_mainWindow.print("Operation (PKI AID) canceled");
        	m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
        	return;
        }
		
		
		// get access privileges
		if (getPKIContainerAccessRules(pkiAID, authTypeByRef) != true)
		{
			m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
			return;
		}
		int keylen = authTypeByRef.keylen;

		// if pin protected ...
		if (authTypeByRef.typeAuth==BSIAccessMethodType.BSI_AM_PIN)
		{
			// AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1
			pinAID = m_mainWindow.showAIDDialog("Enter PIN Applet ID:");
			if (pinAID == null)
	        {
	        	m_mainWindow.print("Operation (Pin AID) canceled");
	        	m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
	        	return;
	        }
			
			if (logIntoCardApplication(pinAID)!=true)
			{
				m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
				return;
			}
			connected = true;
		}
		
		
		//
        // According to RFC 3447 described here: http://www.ietf.org/rfc/rfc3447.txt (section 9.2 EMSA-PKCS1-v1_5), the data to be signed must be formatted as follows:
        // - Padding
        // - Hash algorithm OID
        // - Message hash
        // Original message: Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.
        // MD5 algo is used for signature sample below:
        // - Padding: 00 01 FF  FF 00
        // - Hash algorithm OID: 30 20 30 0c 06 08 2a 86 48 86 f7 0d 02 05 05 00 04 10
        // - Message hash: 81 8c 6e 60 1a 24 f7 27 50 da 0f 6c 9b 8e be 28

		String strData = "3020300c06082a864886f70d020505000410818c6e601a24f72750da0f6c9b8ebe28";
        
        // frame: 00.01.msg.FF...
        byte[] frame = new byte[keylen];
        Arrays.fill(frame, (byte)0xFF);
        frame[0] = 0x00;
        frame[1] = 0x01;
        System.arraycopy(strData.getBytes(), 0, frame, frame.length-strData.length(), strData.length());
        String strFrame = Helper.bytesToHex(frame); 
        m_mainWindow.print("data to be signed (hex): " + Helper.byteArrayToASCIIString(strFrame.getBytes()), LogType.APIInfo);
        byte[] dataToSign = new byte[strFrame.length()];
        System.arraycopy(strFrame.getBytes(), 0, dataToSign, 0, strFrame.length());
        
        
		// pki compute
		short theAlgoId = CryptographicAlgoID.RSA_NO_PAD;
		byte[] theSignature = null;
		try
		{
			theSignature = m_bsi.gscBsiPkiCompute(m_hCard, pkiAID, theAlgoId, dataToSign);
			m_mainWindow.print("Data signature succeeded.");
			m_mainWindow.print("signature (hex): " + Helper.byteArrayToASCIIString(theSignature), LogType.APIInfo);
		}
		catch (Exception ex)
		{
			m_mainWindow.print(ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
			return;
		}
		

		// disconnect
		postamble(connected, pinAID);
		
		m_mainWindow.print("End of Data Sign scenario.", LogType.ScenarioBoundary);
	}
	
	/**
    * Executes the steps of the BSI Read objects scenario.
    *
    */
	public void scenarioReadDataItems()
	{
		String gcAID = "";
		String pinAID = "";
		boolean connected = false;
		reftype authTypeByRef  = new reftype();
		authTypeByRef.typeAuth = 0;
		authTypeByRef.keylen   = -1;
		
		m_mainWindow.print("Starting Read Data Items scenario.", LogType.ScenarioBoundary);
		
		//connect
		m_mainWindow.print("Try connecting to reader ...", LogType.APIInfo);
		if(preamble()!=true)
		{
			m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
			return;
		}
		m_mainWindow.print("m_hCard: " + m_hCard);
		
		// AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1
		gcAID = m_mainWindow.showAIDDialog("Enter GC Applet ID:");
		m_mainWindow.print("GC AID: " + gcAID);
		if (gcAID == null)
        {
        	m_mainWindow.print("Operation (GC AID) canceled");
        	m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
        	return;
        }
		
		// get GC applet properties
		if (getGCContainerAccessRules(gcAID, authTypeByRef) != true)
		{
			m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		// if pin protected ...
		if (authTypeByRef.typeAuth==BSIAccessMethodType.BSI_AM_PIN)
		{
			// AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1
			pinAID = m_mainWindow.showAIDDialog("Enter PIN Applet ID:");
			if (pinAID == null)
	        {
	        	m_mainWindow.print("Operation (Pin AID) canceled");
	        	m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
	        	return;
	        }
			
			if (logIntoCardApplication(pinAID)!=true)
			{
				m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
				return;
			}
			connected = true;
		}
		
		
		m_mainWindow.print("Reading applet's tags ...");
		// read tag list
		short[]	m_taglist = null;
		try {
			m_taglist = m_bsi.gscBsiGcReadTagList(m_hCard, gcAID);
		} catch (BSIException ex) {
			m_mainWindow.print("BSI Exception: " + ex.toString(), LogType.APIInfo);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		if(m_taglist == null)
			m_mainWindow.print("No tags found.");
		else
		{
			m_mainWindow.print("Found "+m_taglist.length+" tags");
		}

		// read tag values
		m_mainWindow.print("Reading tag values:");
		for (int i=0; i<m_taglist.length; i++)
		{
			byte[] tagvalue = null;
			try {
				tagvalue = m_bsi.gscBsiGcReadValue(m_hCard, gcAID, m_taglist[i]);
			} catch (BSIException ex) {
				m_mainWindow.print("Exception: " + ex.toString(), LogType.APIInfo);
				m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
				m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
				return;
			}
			m_mainWindow.print("tag # "+i+" = "+ m_taglist[i]);
			m_mainWindow.print("Value =" + Helper.byteArrayToASCIIString(tagvalue));
		}

		// disconnect
		postamble(connected, pinAID);
		
		m_mainWindow.print("End of Read Data Items scenario.", LogType.ScenarioBoundary);
	}
	
	/**
    * Executes the steps of the BSI send APDUs scenario.
    *
    */
	public void scenarioSendAPDUs()
	{
		String pinAID = "";
		boolean connected = false;
		
		m_mainWindow.print("Starting Send APDUs scenario.", LogType.ScenarioBoundary);
		
		//connect
		m_mainWindow.print("Try connecting to reader ...", LogType.APIInfo);
		if(preamble()!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}
		m_mainWindow.print("m_hCard: " + m_hCard);
		
		// AIDs of applets present on a card can be displayed using the Advanced Diagnostics Tool provided by ActivClient 7.1
		pinAID = m_mainWindow.showAIDDialog("Enter PIN Applet ID:");
		m_mainWindow.print("PIN applet AID: " + pinAID);
		if (pinAID == null)
        {
        	m_mainWindow.print("Operation (PIN AID) canceled");
        	m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
        	return;
        }
		
		if (gscXsiNotifySynchronizationStart()!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		if (bsiUtilBeginTransaction()!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		// select applet
		if (sendAPDUSelectApplet(pinAID)!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		// is pin verified?
		if (sendAPDUPINVerified()!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}

		if (bsiUtilEndTransaction()!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}
				
		if (gscXsiNotifySynchronizationEnd()!=true)
		{
			m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
			return;
		}
		
		// disconnect
		postamble(connected, pinAID);
		
		m_mainWindow.print("End of Send APDUs scenario.", LogType.ScenarioBoundary);
	}
		
	
	
	
	public boolean bsiUtilBeginTransaction()
	{
		boolean res = true;
		boolean blocking = true;  // non-blocking transactions are not supported
		
		try
		{
			if (m_hCard != -1)
			{
				m_bsi.gscBsiUtilBeginTransaction(m_hCard, blocking);
				m_mainWindow.print("Started BSI transaction");
			}
			else
			{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}
	
	public boolean bsiUtilEndTransaction()
	{
		boolean res = true;
		
		try
		{
			if (m_hCard != -1)
			{
				m_bsi.gscBsiUtilEndTransaction(m_hCard);
				m_mainWindow.print("Ended BSI transaction");
			}
			else
			{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}
	
	
	/**
	* Start the notify synchronisation
	* The function calls the method gscXsiNotifySynchronizationStart of GSCBSI
	*/
	public boolean gscXsiNotifySynchronizationStart()
	{
		boolean res = true;

		try
		{
			if (m_hCard != -1)
			{
				m_bsi.gscXsiNotifySynchronizationStart(m_hCard);
				m_mainWindow.print("Started BSI Notify Synchronization");
			}
			else
			{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}

	/**
	* Stop the notify synchronisation
	* The function calls the method gscXsiNotifySynchronizationEnd of GSCBSI
	*/
	public boolean gscXsiNotifySynchronizationEnd()
	{
		boolean res = false;
		
		try
		{
			if (m_hCard != -1)
			{
				m_bsi.gscXsiNotifySynchronizationEnd(m_hCard);
				m_mainWindow.print("Ended BSI Notify Synchronization");
			}
			else
			{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}
	
	
	public boolean sendAPDUSelectApplet(String PINAID)
	{
		boolean res = true;
		
		try
		{
			if (m_hCard != -1)
			{
				String strAPDU = "00A4040007" + PINAID;
				String strAPDUHex = strAPDU; //bytesToHex(strAPDU.getBytes());
				m_mainWindow.print("Send Select Applet APDU = " + strAPDUHex);
				byte[] theResponse = m_bsi.gscBsiUtilPassthru(m_hCard, strAPDUHex.getBytes());
				m_mainWindow.print("Select Applet APDU sent successfully");
				m_mainWindow.print("Response = " + /*Helper.byteArrayToASCIIString(Helper.ConvAsciiToBin(*/Helper.byteArrayToASCIIString(theResponse)/*))*/);
			}
			else{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				return false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : "+ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}
	
	public boolean sendAPDUPINVerified()
	{
		boolean res = true;
		try
		{
			if (m_hCard != -1)
			{
				String strAPDU = "8020000000";
				String strAPDUHex = strAPDU; //bytesToHex(strAPDU.getBytes());
				m_mainWindow.print("Send PIN Verified APDU=" + strAPDUHex);
				byte[] theResponse = m_bsi.gscBsiUtilPassthru(m_hCard, strAPDUHex.getBytes());
				m_mainWindow.print("IsPINVerified APDU sent successfully");
				m_mainWindow.print("Response = " + /*Helper.byteArrayToASCIIString(Helper.ConvAsciiToBin(*/Helper.byteArrayToASCIIString(theResponse)/*))*/);
			}
			else
			{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : "+ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}		
		
		return res;
	}
	
	/**
    * Ends a scenario by terminating the BSI connection : log out of the card if needed, 
    * then disconnect from the reader.
    *
    */
	public void postamble(boolean loggedIn, String pinAID)
	{
		if(loggedIn)
		{
			logoutOfCardApplication(pinAID);
		}
		disconnect();
	}
	

	/**
    * Gets the list of smart card readers (by calling getReaderList), then
    * prompts the user with a popup letting him choose the reader to use.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean preamble()
	{
		listReaders();
		if(m_readerList == null || m_readerList.length == 0)
		{
			m_mainWindow.print("No card reader detected.", LogType.APIInfo);
			return false;
		}
		int choice = m_mainWindow.showSelectionDialog("Select the reader",  m_readerList);
		if(choice == -1 || !connect(m_readerList[choice]))
		{
			return false;
		}
		return true;
	}
	
	
	/**
    * Gets the list of available smart card readers by calling gscBsiUtilGetReaderList.
    *
    */
	public void listReaders()
	{
		
		try 
		{

			Vector<String> m_strReaders = m_bsi.gscBsiUtilGetReaderList();
			if(m_strReaders!=null)
			{
				m_readerList = new String[m_strReaders.size()];
				for(int i=0;i<m_strReaders.size();i++)
				{
					m_readerList[i] = (String) m_strReaders.elementAt(i);
				}
			}
		}
		catch(Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			return;
		}		
	}

	
	/**
    * Connect to the selected card reader by calling gscBsiUtilConnect.
    *
    * @param reader		the name of the card reader to connect to
    */
	public boolean connect(String reader)
	{
		
		m_mainWindow.print("Connect to reader: " + reader);
		
		
		try 
		{
			m_hCard = m_bsi.gscBsiUtilConnect(reader);
			if(m_hCard != -1)
			{
				m_mainWindow.print("Connection to reader succeeded");
			}
			else
			{
				m_mainWindow.print("Failed to connect to card.", LogType.APIError);
				return false;
			}
		}
		catch(Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			return false;
		}
		
		return true;
	}

	
	/**
    * Logs out of the card by calling gscBsiUtilReleaseContext
    *
    */
	public void logoutOfCardApplication(String pinAID)
	{
		
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		try 
		{
			m_bsi.gscBsiUtilReleaseContext(m_hCard, pinAID);
			m_mainWindow.print("Logout succeeded");
		}
		catch(Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
		}
		
	}
	
	/**
    * Disconnects from the selected card reader by calling gscBsiUtilDisconnect.
    *
    */
	public void disconnect()
	{
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		
		try 
		{
			m_bsi.gscBsiUtilDisconnect(m_hCard);
			m_hCard = -1;
			
			m_mainWindow.print("Disconnected from reader");
		}
		catch(Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
		}
	}
	
		
	/**
    * Login to the card. This method asks the user for the PIN of the card, 
    * then calls gscBsiUtilAcquireContext to authenticate him.
    *
    * @return boolean	false if the authentication failed, or the user canceled. 
    * 					If this method returns true, the caller must log out of the card
    * 					when the operations are finished. 
    */
	public boolean logIntoCardApplication(String pinAID)
	{
		BSIAuthenticator myAut;
		Vector<BSIAuthenticator> myAuts = new Vector<BSIAuthenticator>();
		reftype authTypeByRef = new reftype();
		authTypeByRef.typeAuth = 0;
		
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return false;
		}
		
			
		// Build the authenticator array (PIN only). Get the pin
        String sPin = m_mainWindow.showPINDialog();
        if (sPin == null)
        {
        	m_mainWindow.print("Operation (PIN) canceled", LogType.ScenarioError);
        	return false;
        }

		try 
		{
			String pinCodeHex = Helper.bytesToHex(sPin.getBytes());
			m_mainWindow.print("Pin code in hex: " + pinCodeHex);
			myAut = new BSIAuthenticator(BSIAccessMethodType.BSI_AM_PIN,
					1,
					pinCodeHex.getBytes()
					);
			myAuts.add(myAut);						
			m_bsi.gscBsiUtilAcquireContext(m_hCard, pinAID, myAuts);
		}
		catch(Exception ex)
		{
			m_mainWindow.print("BSI Exception : " + ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			return false;
		}
		m_mainWindow.print("Login succeeded");

		return true;
	}

	// encapsulate the authType int
	class reftype { public int typeAuth; public int keylen;}
		
	/**
	* Get the crypto properties for PKI applet.
	* <br>Test getPKIAppletAccessRules(String, reftype) method
	*
	* @see getPKIAppletAccessRules(String, reftype)
	*/
	public boolean getPKIContainerAccessRules(String AID, reftype authRef)
	{
		boolean res = true;
		
		try
		{
			if (m_hCard != -1)
			{
				m_mainWindow.print("Reading PKI Applet ACRs ...");
				
				CryptoProperties theCryptoProp = m_bsi.gscBsiGetCryptoProperties(m_hCard, AID);

				m_mainWindow.print("RSA Key length   = " + theCryptoProp.getKeyLen() );
				authRef.keylen = theCryptoProp.getKeyLen();

				m_mainWindow.print("PKICompute ACR     = "+ACRtoString(theCryptoProp.getCRYPTOacr().getPkiComputeACR().getACRType()));
				if (theCryptoProp.getCRYPTOacr().getPkiComputeACR().getACRType() == BSIAccessControlRules.BSI_ACR_PIN
						|| theCryptoProp.getCRYPTOacr().getPkiComputeACR().getACRType() == BSIAccessControlRules.BSI_ACR_XAUTH_OR_PIN
						|| theCryptoProp.getCRYPTOacr().getPkiComputeACR().getACRType() == BSIAccessControlRules.BSI_ACR_PIN_ALWAYS)
					authRef.typeAuth = BSIAccessMethodType.BSI_AM_PIN;
				else if (theCryptoProp.getCRYPTOacr().getPkiComputeACR().getACRType() != BSIAccessControlRules.BSI_ACR_ALWAYS)
				{
					m_mainWindow.print("Unsupported ACR for data signature", LogType.ScenarioError );
					res = false;
				}
				
			}
			else
			{
				m_mainWindow.print("No BSIConnection", LogType.ScenarioError);
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : "+ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}
	
	/**
	* Converts an Access Control Rule to a string
	*
	* @return string representing the access control rule
	* @param ACR   Access Control Rule to be converted
	*/
	public String ACRtoString(int ACR)
	{
		switch(ACR)
		{
			case BSIAccessControlRules.BSI_ACR_ALWAYS:
				return "BSI_ACR_ALWAYS";
			case BSIAccessControlRules.BSI_ACR_NEVER:
				return "BSI_ACR_NEVER";
			case BSIAccessControlRules.BSI_ACR_XAUTH:
				return "BSI_ACR_XAUTH";
			case BSIAccessControlRules.BSI_ACR_XAUTH_OR_PIN:
				return "BSI_ACR_XAUTH_OR_PIN";
			case BSIAccessControlRules.BSI_SECURE_CHANNEL_GP:
				return "BSI_SECURE_CHANNEL_GP";
			case BSIAccessControlRules.BSI_ACR_PIN_ALWAYS:
				return "BSI_ACR_PIN_ALWAYS";
			case BSIAccessControlRules.BSI_ACR_PIN:
				return "BSI.BSI_ACR_PIN";
			case BSIAccessControlRules.BSI_ACR_XAUTH_THEN_PIN	:
				return "BSI.BSI_ACR_XAUTH_THEN_PIN	";
			case BSIAccessControlRules.BSI_ACR_UPDATE_ONCE:
				return "BSI_ACR_UPDATE_ONCE";
			case BSIAccessControlRules.BSI_ACR_PIN_THEN_XAUTH:
				return "BSI_ACR_PIN_THEN_XAUTH";
			case BSIAccessControlRules.BSI_SECURE_CHANNEL_ISO:
				return "BSI_SECURE_CHANNEL_ISO";
		}
		return "Unknow ACR";
	}
	
	
	/**
	* Gets Access control Rule of the Generic Container.
	* <br>Test GSCBSI.getGCContainerAccessRules(String AID, reftype authRef) method
	*
	* @see GSCBSI#getGCContainerAccessRules(String AID, reftype authRef)
	*/
	public boolean getGCContainerAccessRules(String AID, reftype authRef)
	{
		boolean res = true;
		
		try
		{
			if (m_hCard != -1)
			{

				m_mainWindow.print("Reading GC Applet ACRs ...");
				
				ContainerProperties theProperties = m_bsi.gscBsiGcGetContainerProperties(m_hCard, AID);

				m_mainWindow.print("Read Tag List ACR    = "+ACRtoString(theProperties.getGCacr().getReadTagListACR().getACRType()));
				m_mainWindow.print("Read Tag Value ACR   = "+ACRtoString(theProperties.getGCacr().getReadValueACR().getACRType()));
				if (theProperties.getGCacr().getReadTagListACR().getACRType()==BSIAccessControlRules.BSI_ACR_PIN
						|| theProperties.getGCacr().getReadTagListACR().getACRType()==BSIAccessControlRules.BSI_ACR_PIN_ALWAYS
						|| theProperties.getGCacr().getReadTagListACR().getACRType()==BSIAccessControlRules.BSI_ACR_XAUTH_OR_PIN)
					authRef.typeAuth = BSIAccessMethodType.BSI_AM_PIN;
				else if (theProperties.getGCacr().getReadTagListACR().getACRType() != BSIAccessControlRules.BSI_ACR_ALWAYS)
				{
					m_mainWindow.print("Unsupported ACR for Read Tag List", LogType.ScenarioError);
					res = false;
				}
				
				if (theProperties.getGCacr().getReadValueACR().getACRType()==BSIAccessControlRules.BSI_ACR_PIN
						|| theProperties.getGCacr().getReadValueACR().getACRType()==BSIAccessControlRules.BSI_ACR_PIN_ALWAYS
						|| theProperties.getGCacr().getReadValueACR().getACRType()==BSIAccessControlRules.BSI_ACR_XAUTH_OR_PIN)
					authRef.typeAuth = BSIAccessMethodType.BSI_AM_PIN;
				else if (theProperties.getGCacr().getReadValueACR().getACRType() != BSIAccessControlRules.BSI_ACR_ALWAYS)
				{
					m_mainWindow.print("Unsupported ACR for Read Tag Value", LogType.ScenarioError);
					res = false;
				}
				
			}
			else
			{
				m_mainWindow.print("No BSIConnection");
				res = false;
			}
		}
		catch (Exception ex)
		{
			m_mainWindow.print("BSI Exception : "+ex.toString(), LogType.APIError);
			m_mainWindow.print(gscBsiUtilGetExtendedErrorText(), LogType.APIError);
			res = false;
		}
		
		return res;
	}
	
	public String gscBsiUtilGetExtendedErrorText()
	{
		String msg = "";
		try
		{
			msg =m_bsi.gscBsiUtilGetExtendedErrorText(m_hCard);
		}
		catch (Exception ex)
		{
			m_mainWindow.print("Exception : "+ex.toString(), LogType.APIError);
		}
		
		return msg;
	}
	
	
	
	@Override
	public void runScenario(int scenarioIdx)
	{
		switch(scenarioIdx)
		{
		case 0:
			scenarioSignData();
			break;
		case 1:
			scenarioReadDataItems();
			break;
		case 2:
			scenarioSendAPDUs();
			break;
			
		default:
			m_mainWindow.print("Combo box state corrupt", LogType.ScenarioError);
			break;
		}
	}
}
