/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */

package ui;

import javax.swing.*;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;

import java.awt.*;

// ---------------------------------------------------------------------------------
//	Class MainWindow
//
//	Main UI class.
//---------------------------------------------------------------------------------

public class MainWindow implements Runnable
{
	//	Graphical elements of this sample.
	private JFrame myFrame;
    private JComboBox<String> m_comboBox;
	private JTextPane m_textPane;
	private JButton m_executeButton;
	private JLabel m_scenarioLabel;
	private JButton m_clearButton;
	private JButton m_quitButton;
	private JScrollPane m_scrollPane;
	
	private String m_windowTitle;
	
	private int m_frameWidth = 600;
	private int m_frameHeight = 600;
	
	private ScenarioInterface m_scenarios;
	
	/* windowAdapter to close the Frame window */
	class SymWindow extends java.awt.event.WindowAdapter
	{
		public void windowClosing(java.awt.event.WindowEvent event)
		{
			Object object = event.getSource();
			if (object == myFrame)
				myFrame.dispose();
		}
	}
	
	// The different types of logs, used for text formatting
	public enum LogType
	{
		ScenarioBoundary,
        ScenarioInfo,
        ScenarioError,
        APIInfo,
        APIError,
        Default
	}
	
	//	Constructor
	public MainWindow(ScenarioInterface scenarioInterface, String [] scenarioList, String windowTitle)
    {
		m_scenarios = scenarioInterface;
		m_windowTitle = windowTitle;

        try
		{
			// Change the look and feel to be closer to the system one
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		}
		catch (ClassNotFoundException | InstantiationException
				| IllegalAccessException | UnsupportedLookAndFeelException e1)
		{
			e1.printStackTrace();
		}
        m_textPane = new JTextPane();
    	m_textPane.setContentType("text/html");
        m_comboBox = new JComboBox<>(scenarioList);
        
        SwingUtilities.invokeLater(this);
    }
	
	/**
    * Clears the text area.
    *
    */
	public void clearOutput()
    {
        m_textPane.setText("");
    }
	
	/**
    * Initializes all the widgets of the main window.
    *
    */
	protected void setupUI()
	{
		GridBagConstraints c = new GridBagConstraints();
		int insUnit = 5;
		
		m_scrollPane = new JScrollPane(m_textPane);
		m_scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		m_textPane.setEditable(false);
		m_textPane.setPreferredSize(new Dimension(100, 100));
		
		c.fill = GridBagConstraints.BOTH;
		
		c.weightx = 0.5;
	    c.gridx = 0;
	    c.gridy = 0;
	    c.gridwidth = 1;
	    c.weighty = 1.0;
	    c.insets = new Insets(insUnit, insUnit, insUnit, insUnit);
	    myFrame.add(m_scenarioLabel, c);
	    
	    c.weightx = 5;
	    c.gridx = 1;
	    c.gridy = 0;
	    c.gridwidth = 1;
	    c.weighty = 1.0;
	    c.insets = new Insets(insUnit, 0, insUnit, insUnit);
	    myFrame.add(m_comboBox, c);
	    
	    c.weightx = 1;
	    c.gridx = 3;
	    c.gridy = 0;
	    c.gridwidth = 1;
	    c.weighty = 1.0;
	    c.insets = new Insets(insUnit, 0, insUnit, insUnit);
	    myFrame.add(m_executeButton, c);
	    
	    c.weightx = 1;
	    c.gridx = 0;
	    c.gridy = 1;
	    c.gridwidth = 4;
	    c.weighty = 100.0;
	    c.insets = new Insets(0, insUnit, insUnit, insUnit);
	    myFrame.add(m_scrollPane, c);
	    
	    c.weightx = 1;
	    c.gridx = 3;
	    c.gridy = 2;
	    c.gridwidth = 1;
	    c.weighty = 1.0;
	    c.insets = new Insets(0, 0, insUnit, insUnit);
	    myFrame.add(m_clearButton, c);
	    
	    c.weightx = 1;
	    c.gridx = 3;
	    c.gridy = 3;
	    c.gridwidth = 1;
	    c.weighty = 1.0;
	    c.insets = new Insets(0, 0, insUnit, insUnit);
	    myFrame.add(m_quitButton, c);
	    
	    //	Set the menu callback functions.
	    SymAction lSymAction = new SymAction();
	    m_executeButton.addActionListener(lSymAction);
	    m_clearButton.addActionListener(lSymAction);
	    m_quitButton.addActionListener(lSymAction);
	}

	/* adapter to catch menu actions */
	class SymAction implements java.awt.event.ActionListener
	{
		public void actionPerformed(java.awt.event.ActionEvent event)
		{
			Object object = event.getSource();

			if (object == m_executeButton)
			{
				m_scenarios.runScenario(m_comboBox.getSelectedIndex());
			}
			else if (object == m_clearButton)
			{
				clearOutput();
			}
			else if (object == m_quitButton)
			{
				print("Exiting...");	
				System.exit(0);
			}
		}
	} // end class SymAction
	
	
    
	/**
	    * Initializes the GUI
	    *
	    */
    @Override
	public void run()
	{
    	myFrame = new JFrame(m_windowTitle);
    	m_executeButton = new JButton("Execute");
    	m_scenarioLabel = new JLabel("Scenario:");
    	m_clearButton = new JButton("Clear logs");
    	m_quitButton = new JButton("Quit");
    	
    	
    	
		myFrame.setLayout(new GridBagLayout());
		myFrame.setSize(m_frameWidth, m_frameHeight);
		myFrame.setMinimumSize(new Dimension(300, 300));
		
		Toolkit tk = Toolkit.getDefaultToolkit();
        Dimension d = tk.getScreenSize();
        int screenHeight = d.height;
        int screenWidth = d.width;
        myFrame.setLocation((screenWidth-m_frameWidth)/2, (screenHeight-m_frameHeight)/3);

		SymWindow aSymWindow = new SymWindow();
		myFrame.addWindowListener(aSymWindow);
	
		//	Create the application menu.
		setupUI();
		
		myFrame.setVisible(true);
	}
    
    /**
     * Appends a message to the text area.
     * 
     * @param msg 	the message to print
     *
     */
	public void print(String msg)
	{
		print(msg, LogType.Default);
	}
	
	/**
    * Appends a message to the text area.
    * 
    * @param msg 	the message to print
    * @param type	the type of message to print, that determines the formatting of the output text
    *
    */
	public void print(String msg, LogType type)
	{
		System.out.println(msg);
        Document doc = m_textPane.getDocument();
        SimpleAttributeSet attributes = new SimpleAttributeSet();
        try
		{
	        switch(type)
	        {
	        case ScenarioBoundary:
	    	    StyleConstants.setBold(attributes, true);
				doc.insertString(doc.getLength(), msg+"\n", attributes); 
	        	break;
	        case ScenarioError:
	        	StyleConstants.setForeground(attributes, new Color(127, 0, 0));
				doc.insertString(doc.getLength(), msg+"\n", attributes);
	        	break;
	        case APIInfo:
	        	StyleConstants.setForeground(attributes, new Color(0, 127, 0));
				doc.insertString(doc.getLength(), "\t"+msg+"\n", attributes);
	        	break;
	        case APIError:
	        	StyleConstants.setForeground(attributes, new Color(127, 0, 0));
				doc.insertString(doc.getLength(), "\t"+msg+"\n", attributes);
	        	break;
	        case ScenarioInfo:
	        case Default:
	    	default:
	    		doc.insertString(doc.getLength(), msg+"\n", attributes);
	    		break;
	        }
		}
        catch (BadLocationException e)
		{
			e.printStackTrace();
		}
	}

	public int showSelectionDialog(String title, String [] choices)
	{
		//"Select the reader"
		SelectionDialog theDlg = new SelectionDialog(myFrame, title,  choices);
		if (!theDlg.isOk() || theDlg.m_sSelection.length() == 0)
		{
			return -1;
		}
		return theDlg.m_selectionIdx;
	}

	public String showPINDialog()
	{
		PINDialog theDlg = new PINDialog(myFrame, "Enter the PIN code", null);
		if(theDlg.m_bCanceled)
		{
			return null;
		}
		return theDlg.getString();
	}

	public void showMessageDialog(String popupMsg)
	{
		JOptionPane.showMessageDialog(myFrame, popupMsg, m_windowTitle, JOptionPane.INFORMATION_MESSAGE);
	}
    
	
	/**
	* Gets applet Identifier from a dialog box.
	* The function stores the AID in the class member m_AID. 
	*/
	public String showAIDDialog(String msg)
	{
		AIDDialog theDlg = new AIDDialog(myFrame, msg, null);
		if(theDlg.m_bCanceled)
		{
			return null;
		}
		
		return theDlg.getString();
	}
} // end class MainWindow


