/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */
 
#include "stdafx.h"

#include "Scenarios.h"
#include "Scenarios.UI.h"
#include "Helper.h"

const CERTIFICATE_INFO g_certificateInfoTable[] = {
	{OBJECT_LABEL_X509_PIV_AUTHENTICATION,			OBJECT_OID_X509_PIV_AUTHENTICATION,			KEYREFERENCEID_PIV_AUTHENTICATION,		SECURITY_CONDITION_PIN},
	{OBJECT_LABEL_X509_DIGITAL_SIGNATURE,			OBJECT_OID_X509_DIGITAL_SIGNATURE,			KEYREFERENCEID_SIGNATURE,				SECURITY_CONDITION_PIN_ALWAYS},
	{OBJECT_LABEL_X509_CARD_AUTHENTICATION,			OBJECT_OID_X509_CARD_AUTHENTICATION,		KEYREFERENCEID_CARD_AUTHENTICATION,		SECURITY_CONDITION_ALWAYS}
};

const OBJECT_INFO g_objectInfoTable[] = {
	{OBJECT_LABEL_CARD_COMPABILITY_CONTAINER,		OBJECT_OID_CARD_COMPABILITY_CONTAINER,		ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_CARD_HOLDER_UNIQUE_IDENTIFIER,	OBJECT_OID_CARD_HOLDER_UNIQUE_IDENTIFIER,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_X509_PIV_AUTHENTICATION,			OBJECT_OID_X509_PIV_AUTHENTICATION,			ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_CARDHOLDER_FINGERPRINTS,			OBJECT_OID_CARDHOLDER_FINGERPRINTS,			ACCESS_RULE_PIN},
	{OBJECT_LABEL_SECURITY_OBJECT,					OBJECT_OID_SECURITY_OBJECT,					ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_CARDHOLDER_FACIAL_IMAGE,			OBJECT_OID_CARDHOLDER_FACIAL_IMAGE,			ACCESS_RULE_PIN},
	{OBJECT_LABEL_PRINTED_INFORMATION,				OBJECT_OID_PRINTED_INFORMATION,				ACCESS_RULE_PIN},
	{OBJECT_LABEL_X509_DIGITAL_SIGNATURE,			OBJECT_OID_X509_DIGITAL_SIGNATURE,			ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_X509_KEY_MANAGEMENT,				OBJECT_OID_X509_KEY_MANAGEMENT,				ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_X509_CARD_AUTHENTICATION,			OBJECT_OID_X509_CARD_AUTHENTICATION,		ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_DISCOVERY_OBJECT,					OBJECT_OID_DISCOVERY_OBJECT,				ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_KEY_HISTORY_OBJECT,				OBJECT_OID_KEY_HISTORY_OBJECT,				ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_1,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_1,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_2,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_2,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_3,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_3,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_4,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_4,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_5,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_5,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_6,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_6,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_7,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_7,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_8,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_8,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_9,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_9,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_10,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_10,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_11,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_11,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_12,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_12,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_13,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_13,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_14,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_14,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_15,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_15,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_16,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_16,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_17,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_17,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_18,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_18,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_19,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_19,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_RETIRED_X509_KEY_MANAGEMENT_20,	OBJECT_OID_RETIRED_X509_KEY_MANAGEMENT_20,	ACCESS_RULE_ALWAYS},
	{OBJECT_LABEL_CARDHOLDER_IRIS_IMAGES,			OBJECT_OID_CARDHOLDER_IRIS_IMAGES,			ACCESS_RULE_ALWAYS}
};

//
// According to RFC 3447 described here: http://www.ietf.org/rfc/rfc3447.txt (section 9.2 EMSA-PKCS1-v1_5), the data to be signed must be formatted as follows:
// - Padding
// - Hash algorithm OID
// - Message hash
// Original message: Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.
// MD5 algo is used for signature sample below:
// - Padding: 00 01 FF  FF 00
// - Hash algorithm OID: 30 20 30 0c 06 08 2a 86 48 86 f7 0d 02 05 05 00 04 10
// - Message hash: 81 8c 6e 60 1a 24 f7 27 50 da 0f 6c 9b 8e be 28
const BYTE g_ucDataToSign[] = {	0x30, 0x20, 0x30, 0x0C, 0x06, 0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x02, 0x05, 0x05, 0x00, 0x04, 0x10, \
								0x81, 0x8c, 0x6e, 0x60, 0x1a, 0x24, 0xf7, 0x27, 0x50, 0xda, 0x0f, 0x6c, 0x9b, 0x8e, 0xbe, 0x28
							};


BOOL Preamble(PIV_CARDHANDLE* cardHandle)
{
	BOOL		bRet = FALSE;
	DWORD		dwRet = ERROR_SUCCESS;
	CHAR		szSelectedReaderName[BUFSIZ];
	PIV_RV		pivRet = PIV_OK;

	if (NULL == cardHandle)
	{
		wprintf_s (L"Preamble: bad parameter.");
		wprintf_s(L"\n");
		goto end_function;
	}

	// Show the reader list and return the seleted one
	if (TRUE == GetSelectedReader(szSelectedReaderName, BUFSIZ))
	{
		// Connect to the smart card inserted in the selected reader
		dwRet = Connect(&pivRet, cardHandle, szSelectedReaderName, (DWORD)strlen(szSelectedReaderName)+1);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivConnect failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
			goto end_function;
		}

		// Select the PIV application
		dwRet = SelectCardApplication(&pivRet, *cardHandle);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivSelectCardApplication failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
			goto end_function;
		}

		bRet = TRUE;
	}

end_function:
	return bRet;
}


BOOL Postamble(PIV_CARDHANDLE* cardHandle, LPBOOL pbLoggedIn)
{
	BOOL		bRet = FALSE;
	DWORD		dwRet = ERROR_SUCCESS;
	PIV_RV		pivRet = PIV_OK;

	if ((NULL == cardHandle) || (NULL == pbLoggedIn))
	{
		wprintf_s (L"Postamble: bad parameter.");
		wprintf_s(L"\n");
		goto end_function;
	}

	// Log out if needed
	if (TRUE == *pbLoggedIn)
	{
		dwRet = LogoutOfCardApplication(&pivRet, *cardHandle);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivLogoutOfCardApplication failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
		}

		*pbLoggedIn = FALSE;
	}

	// Disconnect from the smart card if needed
	if (INVALID_CARD_HANDLE != *cardHandle)
	{
		dwRet = Disconnect(&pivRet, *cardHandle);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivDisconnect failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
		}
		*cardHandle = INVALID_CARD_HANDLE;
	}

	bRet = TRUE;

end_function:
	return bRet;
}


BOOL PIVSignature()
{
	BOOL				bRet = FALSE;
	DWORD				dwRet = ERROR_SUCCESS;
	PIV_RV				pivRet = PIV_OK;
	PIV_CARDHANDLE		cardHandle = INVALID_CARD_HANDLE;
	BOOL				bLoggedIn = FALSE;
	CERTIFICATE_INDEX	nSelectedCertificateIndex = CERTIFICATE_INDEX_NO_CERTIFICATE;
	SECURITY_CONDITION	currentSecurityCondition = SECURITY_CONDITION_ALWAYS;
	PIV_Byte			keyReference = 0;
	LPBYTE				ucSignature = NULL;
	DWORD				cbSignature = 0;

	// Exectute the basic steps for further operations
	if (FALSE == Preamble(&cardHandle))
	{
		goto end_function;
	}

	// Show the certificates list and return the seleted one
	nSelectedCertificateIndex = GetSelectedCertificate(cardHandle);
	if (CERTIFICATE_INDEX_NO_CERTIFICATE == nSelectedCertificateIndex)
	{
		goto end_function;
	}

	// Get the selected certificate security condition for use and key reference
	currentSecurityCondition = g_certificateInfoTable[nSelectedCertificateIndex].securityCondition;
	keyReference = g_certificateInfoTable[nSelectedCertificateIndex].keyReference;

	// Perform a login if required
	if ((SECURITY_CONDITION_PIN == currentSecurityCondition) || (SECURITY_CONDITION_PIN_ALWAYS == currentSecurityCondition))
	{
		BYTE		ucPIN[16];
		DWORD		cbPIN = _countof(ucPIN);

		DisplayPINInput(ucPIN, &cbPIN);

		dwRet = LogIntoCardApplication(&pivRet, cardHandle, ucPIN, cbPIN);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivLogIntoCardApplication failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
			goto end_function;
		}

		bLoggedIn = TRUE;
	}

	// Compute the signature buffer size
	dwRet = SignData(&pivRet, cardHandle, g_ucDataToSign, _countof(g_ucDataToSign), keyReference, ucSignature, &cbSignature);
	if (ERROR_INSUFFICIENT_BUFFER != dwRet)
	{
		if (ERROR_PIV_API == dwRet)
		{
			wprintf(L"pivCrypt failed with error %s", GetPivErrorString(pivRet));
			wprintf_s(L"\n");
		}
		goto end_function;
	}

	// Allocate the signature buffer
	ucSignature = (LPBYTE)calloc(cbSignature, sizeof(BYTE));
	if (NULL == ucSignature)
	{
		wprintf(L"Memory allocation error");
		wprintf_s(L"\n");
		goto end_function;
	}

	// Sign data
	dwRet = SignData(&pivRet, cardHandle, g_ucDataToSign, _countof(g_ucDataToSign), keyReference, ucSignature, &cbSignature);
	if (ERROR_SUCCESS != dwRet)
	{
		if (ERROR_PIV_API == dwRet)
		{
			wprintf(L"pivCrypt failed with error %s", GetPivErrorString(pivRet));
			wprintf_s(L"\n");
		}
		goto end_function;
	}

	// Display the signature
	wprintf_s(L"\n");
	wprintf_s(L"Signature:");
	wprintf_s(L"\n");
	DisplayBinInHex(ucSignature, cbSignature);
	wprintf_s(L"\n");
	wprintf_s(L"\n");

	// Reset the signature buffer
	free(ucSignature);
	ucSignature = NULL;
	cbSignature = 0;

	// Perform a second login if required
	if (SECURITY_CONDITION_PIN_ALWAYS == currentSecurityCondition)
	{
		BYTE		ucPIN[16];
		DWORD		cbPIN = _countof(ucPIN);

		DisplayPINInput(ucPIN, &cbPIN);

		dwRet = LogIntoCardApplication(&pivRet, cardHandle, ucPIN, cbPIN);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivLogIntoCardApplication failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
			goto end_function;
		}

		bLoggedIn = TRUE;
	}

	// Compute the signature buffer size
	dwRet = SignData(&pivRet, cardHandle, g_ucDataToSign, _countof(g_ucDataToSign), keyReference, ucSignature, &cbSignature);
	if (ERROR_INSUFFICIENT_BUFFER != dwRet)
	{
		if (ERROR_PIV_API == dwRet)
		{
			wprintf(L"pivCrypt failed with error %s", GetPivErrorString(pivRet));
			wprintf_s(L"\n");
		}
	}

	// Allocate the signature buffer
	ucSignature = (LPBYTE)calloc(cbSignature, sizeof(BYTE));
	if (NULL == ucSignature)
	{
		wprintf(L"Memory allocation error");
		wprintf_s(L"\n");
		goto end_function;
	}

	// Sign data
	dwRet = SignData(&pivRet, cardHandle, g_ucDataToSign, _countof(g_ucDataToSign), keyReference, ucSignature, &cbSignature);
	if (ERROR_SUCCESS != dwRet)
	{
		if (ERROR_PIV_API == dwRet)
		{
			wprintf(L"pivCrypt failed with error %s", GetPivErrorString(pivRet));
			wprintf_s(L"\n");
		}
		goto end_function;
	}


	// Display the signature
	wprintf_s(L"Signature:");
	wprintf_s(L"\n");
	DisplayBinInHex(ucSignature, cbSignature);

	bRet = TRUE;

end_function:
	if (ucSignature)
	{
		free(ucSignature);
		ucSignature = NULL;
	}

	// Execute the common finalization steps
	if (FALSE == Postamble(&cardHandle, &bLoggedIn))
	{
		bRet = FALSE;
	}

	return bRet;
}

BOOL PIVObjectsRead()
{
	BOOL				bRet = FALSE;
	DWORD				dwRet = ERROR_SUCCESS;
	PIV_RV				pivRet = PIV_OK;
	PIV_CARDHANDLE		cardHandle = INVALID_CARD_HANDLE;
	BOOL				bLoggedIn = FALSE;
	OBJECT_INDEX		nSelectedObjectIndex = OBJECT_INDEX_NO_OBJECT;
	ACCESS_RULE			currentAccessRule = ACCESS_RULE_ALWAYS;
	LPBYTE				ucObjectData = NULL;
	DWORD				cbObjectData = 0;

	// Exectute the basic steps for further operations
	if (FALSE == Preamble(&cardHandle))
	{
		goto end_function;
	}

	// Show the piv objects list and return the seleted one
	nSelectedObjectIndex = GetSelectedObject(cardHandle);
	if (OBJECT_INDEX_NO_OBJECT == nSelectedObjectIndex)
	{
		goto end_function;
	}

	// Get the selected piv object access rule
	currentAccessRule = g_objectInfoTable[nSelectedObjectIndex].accessRule;

	// Perform a login if required
	if (ACCESS_RULE_PIN == currentAccessRule)
	{
		BYTE		ucPIN[16];
		DWORD		cbPIN = _countof(ucPIN);

		DisplayPINInput(ucPIN, &cbPIN);

		dwRet = LogIntoCardApplication(&pivRet, cardHandle, ucPIN, cbPIN);
		if (ERROR_SUCCESS != dwRet)
		{
			if (ERROR_PIV_API == dwRet)
			{
				wprintf(L"pivLogIntoCardApplication failed with error %s", GetPivErrorString(pivRet));
				wprintf_s(L"\n");
			}
			goto end_function;
		}

		bLoggedIn = TRUE;
	}

	// Compute the piv object buffer size
	dwRet = ReadObject(&pivRet, cardHandle, nSelectedObjectIndex, ucObjectData, &cbObjectData);
	if (ERROR_INSUFFICIENT_BUFFER != dwRet)
	{
		if (ERROR_PIV_API == dwRet)
		{
			wprintf(L"pivGetData failed with error %s", GetPivErrorString(pivRet));
			wprintf_s(L"\n");
		}
		goto end_function;
	}

	// Piv objects could be empty
	if (0 == cbObjectData)
	{
		wprintf_s(L"Empty piv object.");
		bRet = TRUE;
		goto end_function;
	}

	// Allocate the piv object buffer size
	ucObjectData = (LPBYTE)calloc(cbObjectData, sizeof(BYTE));
	if (NULL == ucObjectData)
	{
		wprintf(L"Memory allocation error");
		wprintf_s(L"\n");
		goto end_function;
	}

	// Read the piv object data
	dwRet = ReadObject(&pivRet, cardHandle, nSelectedObjectIndex, ucObjectData, &cbObjectData);
	if (ERROR_SUCCESS != dwRet)
	{
		if (ERROR_PIV_API == dwRet)
		{
			wprintf(L"pivGetData failed with error %s", GetPivErrorString(pivRet));
			wprintf_s(L"\n");
		}
		goto end_function;
	}

	wprintf_s(L"Object data:");
	wprintf_s(L"\n");
	DisplayBinInHex(ucObjectData, cbObjectData);

	bRet = TRUE;

end_function:
	if (ucObjectData)
	{
		free(ucObjectData);
		ucObjectData = NULL;
	}

	// Execute the common finalization steps
	if (FALSE == Postamble(&cardHandle, &bLoggedIn))
	{
		bRet = FALSE;
	}

	return bRet;
}


DWORD ListReaders(LPSTR szReadersList, PDWORD pcbReadersList)
{
	DWORD			dwRet = ERROR_SUCCESS;
	PIV_RV			pivRet = PIV_OK;

	// Connection Description Template so use tag '7F21'
	PIV_Byte		ucConnection[7] = {
		0x7f,
		0x21,
		0x04, // Indicates the CDT length
		0x81, // Tag corresponding to the 'Card reader name'
		0x00, // Indicates we want the readers list
		0x90, // Tag corresponding to 'Network node - Local'
		0x00  // Indicates the length for the preceding tag
	};

	// Parameters
	PIV_Byte		connectionDescription[5000]; // Reserve enough space to get the readers names list
	PIV_ULong32		connectionDescriptionLength = 5000;
	PIV_CARDHANDLE	cardHandle = INVALID_CARD_HANDLE;

	if (NULL == pcbReadersList)
	{
		wprintf_s (L"ListReaders: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	// Initialize the IN/OUT buffer with the Connection Description Template.
	memcpy(connectionDescription, ucConnection, _countof(ucConnection));

	pivRet = pivConnect(
		TRUE,
		connectionDescription,
		&connectionDescriptionLength,
		&cardHandle);

	if (PIV_OK != pivRet)
	{
		*pcbReadersList = 0;
		goto end_function;
	}

	if (0 == connectionDescriptionLength)
	{
		*pcbReadersList = connectionDescriptionLength;
		goto end_function;
	}

	if (connectionDescriptionLength > (*pcbReadersList))
	{
		*pcbReadersList = connectionDescriptionLength;
		dwRet = ERROR_INSUFFICIENT_BUFFER;
		goto end_function;
	}

	if (NULL != szReadersList)
	{
		memcpy_s(szReadersList, *pcbReadersList, connectionDescription, connectionDescriptionLength);
	}

	*pcbReadersList = connectionDescriptionLength;

end_function:
	if (INVALID_CARD_HANDLE != cardHandle)
	{
		pivDisconnect(cardHandle);
	}

	return dwRet;
}


DWORD Connect(PIV_RV* pivRet, PIV_CARDHANDLE* cardHandle, LPCSTR szReader, const DWORD cbReader)
{
	DWORD			dwRet = ERROR_SUCCESS;

	// Parameters
	PIV_ULong32		connectionDescriptionLength = 0;
	PIV_Byte*		connectionDescription = NULL;

	if ((NULL == pivRet) || (NULL == cardHandle) || (NULL == szReader) || (0 == cbReader))
	{
		wprintf_s (L"Connect: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	//	Build connection description
	connectionDescriptionLength = 7 + (PIV_ULong32)cbReader-1; //We have to store 7 bytes and the reader name without null terminating character
	connectionDescription = (PIV_Byte*)malloc(connectionDescriptionLength);
	if (NULL == connectionDescription)
	{
		wprintf_s (L"Connect: allocation error.");

		dwRet = ERROR_OUTOFMEMORY;
		goto end_function;
	}

	// Connection Description Template
	connectionDescription[0] = CONNECTIONDESCRIPTIONTAG_INITIALIZE_1;
	connectionDescription[1] = CONNECTIONDESCRIPTIONTAG_INITIALIZE_2;
	connectionDescription[2] = (PIV_Byte)connectionDescriptionLength-3; // Indicates the CDT length. Works only with reader names below 123 bytes long.
	connectionDescription[3] = CONNECTIONDESCRIPTIONTAG_CARD_READER_NAME;
	connectionDescription[4] = (PIV_Byte)cbReader-1; // Indicates the length for the preceding tag
	memcpy(connectionDescription+5, szReader, cbReader-1); // Paste the reader name
	connectionDescription[connectionDescriptionLength-2] = CONNECTIONDESCRIPTIONTAG_NETWORK_NODE_LOCAL;
	connectionDescription[connectionDescriptionLength-1] = 0x00; // Indicates the length for the preceding tag

	// Let's really try to connect
	*pivRet = pivConnect(
		TRUE,
		connectionDescription,
		&connectionDescriptionLength,
		cardHandle);

	if (PIV_OK != (*pivRet))
	{
		*cardHandle = INVALID_CARD_HANDLE;

		dwRet = ERROR_PIV_API;
	}

end_function:
	if (NULL != connectionDescription)
	{
		free(connectionDescription);
	}

	return dwRet;
}


DWORD Disconnect(PIV_RV* pivRet, PIV_CARDHANDLE cardHandle)
{
	DWORD			dwRet = ERROR_SUCCESS;

	if ((NULL == pivRet) || (INVALID_CARD_HANDLE == cardHandle))
	{
		wprintf_s (L"Disconnect: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	*pivRet = pivDisconnect(cardHandle);

	if (PIV_OK != (*pivRet))
	{
		dwRet = ERROR_PIV_API;
	}

end_function:
	return dwRet;
}


DWORD SelectCardApplication(PIV_RV* pivRet, const PIV_CARDHANDLE cardHandle)
{
	DWORD			dwRet = ERROR_SUCCESS;

	// Parameters
	PIV_Byte		applicationAID[] = PIV_CARD_APPLICATION_AID_RIGHTTRONCATED;
	PIV_ULong32		applicationAIDLength = sizeof(applicationAID);
	PIV_Byte		applicationProperties[BUFSIZ];
	PIV_ULong32		applicationPropertiesLength = sizeof(applicationProperties);

	if ((NULL == pivRet) || (INVALID_CARD_HANDLE == cardHandle))
	{
		wprintf_s (L"SelectCardApplication: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	*pivRet = pivSelectCardApplication(cardHandle,
		applicationAID,
		applicationAIDLength,
		applicationProperties,
		&applicationPropertiesLength);

	if (PIV_OK != (*pivRet))
	{
		dwRet = ERROR_PIV_API;
	}

end_function:
	return dwRet;
}


DWORD LogIntoCardApplication(PIV_RV* pivRet, const PIV_CARDHANDLE cardHandle, LPBYTE ucPIN, const DWORD cbPIN)
{
	DWORD			dwRet = ERROR_SUCCESS;

	// Parameters
	PIV_Byte*		authenticators		= NULL;
	PIV_ULong32		authenticatorsLength = 0;

	if ((NULL == pivRet) || (INVALID_CARD_HANDLE == cardHandle) || (NULL == ucPIN) || (0 == cbPIN))
	{
		wprintf_s (L"LogIntoCardApplication: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	// Use PIN authentication.
	authenticatorsLength = 2 + 2 + cbPIN + 3;
	authenticators = (PIV_Byte*)malloc(authenticatorsLength);
	if (NULL == authenticators)
	{
		wprintf_s (L"LogIntoCardApplication: allocation error.");

		dwRet = ERROR_OUTOFMEMORY;
		goto end_function;
	}

	// Authenticator Template so use tag '67'
	authenticators[0] = 0x67;
	authenticators[1] = (PIV_Byte) authenticatorsLength-2; // Indicates the AT length
	authenticators[2] = 0x81; // Reference data (Indicating PIN value)
	authenticators[3] = (PIV_Byte) cbPIN; // Indicates the length for the preceding tag
	memcpy(authenticators+4, ucPIN, cbPIN); // Paste the PIN value as MBCS array
	authenticators[4+cbPIN] = 0x83; // Key reference
	authenticators[5+cbPIN] = 0x01; // Indicates the length for the preceding tag
	authenticators[6+cbPIN] = 0x80; // Value of the key reference: application PIN

	*pivRet = pivLogIntoCardApplication(cardHandle,
		authenticators,
		authenticatorsLength);

	if (PIV_OK != (*pivRet))
	{
		dwRet = ERROR_PIV_API;
	}

end_function:
	if (NULL != authenticators)
	{
		free(authenticators);
	}

	return dwRet;
}


DWORD LogoutOfCardApplication(PIV_RV* pivRet, const PIV_CARDHANDLE cardHandle)
{
	DWORD			dwRet = ERROR_SUCCESS;

	if ((NULL == pivRet) || (INVALID_CARD_HANDLE == cardHandle))
	{
		wprintf_s (L"LogoutOfCardApplication: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	*pivRet = pivLogoutOfCardApplication(cardHandle);

	if (PIV_OK != (*pivRet))
	{
		dwRet = ERROR_PIV_API;
	}

end_function:
	return dwRet;
}


DWORD ListCertificates(const PIV_CARDHANDLE cardHandle, CERTIFICATE_INDEX* pCertificates, PDWORD pcbCertificates)
{
	DWORD			dwRet = ERROR_SUCCESS;
	CERTIFICATE_INDEX		nCertIndex = CERTIFICATE_INDEX_NO_CERTIFICATE;
	CERTIFICATE_INDEX		certificates[CERTIFICATE_INDEX_CARD_AUTHENTICATION+1];
	LPSTR			pCertificateOID = NULL;
	DWORD			wMenuIndex = 0;
	int				i = 0;
	PIV_RV			pivRet = PIV_OK;

	// Parameters
	PIV_Byte		oid[BUFSIZ];
	PIV_ULong32		oidLength = 0;
	PIV_ULong32		dataLength = 0;

	if ((INVALID_CARD_HANDLE == cardHandle) || (NULL == pcbCertificates))
	{
		wprintf_s (L"ListCertificates: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	// Initialize array to NO_CERTIFICATE
	for (i=0; i<_countof(certificates); i++)
	{
		certificates[i] = CERTIFICATE_INDEX_NO_CERTIFICATE;
	}

	for (nCertIndex=CERTIFICATE_INDEX_PIV_AUTHENTICATION; nCertIndex<=CERTIFICATE_INDEX_CARD_AUTHENTICATION; nCertIndex++)
	{
		pCertificateOID = g_certificateInfoTable[nCertIndex].szCertOID;

		memcpy_s(oid, sizeof(oid), pCertificateOID, strlen(pCertificateOID));
		oidLength = (PIV_ULong32)strlen(pCertificateOID);
		dataLength = 0;

		pivRet = pivGetData(
			cardHandle,
			oid,
			oidLength,
			NULL,
			&dataLength);

		if ((PIV_INSUFFICIENT_BUFFER == pivRet) && (dataLength > 0))
		{
			// Add available and initialized certificates to the array
			certificates[wMenuIndex++] = nCertIndex;
		}
	}

	if (0 == wMenuIndex)
	{
		*pcbCertificates = wMenuIndex;
		goto end_function;
	}

	if ((*pcbCertificates) < wMenuIndex)
	{
		*pcbCertificates = wMenuIndex;
		dwRet = ERROR_INSUFFICIENT_BUFFER;
		goto end_function;
	}

	if (NULL != pCertificates)
	{
		// Return the certificates array
		memcpy_s(pCertificates, (*pcbCertificates)*sizeof(CERTIFICATE_INDEX), certificates, wMenuIndex*sizeof(CERTIFICATE_INDEX));
	}

	// Return the number of elements
	*pcbCertificates = wMenuIndex;

end_function:
	return dwRet;
}



DWORD SignData(PIV_RV* pivRet, const PIV_CARDHANDLE cardHandle, LPCBYTE ucDataToSign, const DWORD cbDataToSign, PIV_Byte keyReference, LPBYTE ucSignature, PDWORD pcbSignature)
{
	DWORD			dwRet = ERROR_SUCCESS;
	LPBYTE			pPaddedAlgoInput = NULL;
	UINT			unPaddedAlgoInputLen = 0;

	// Parameters
	PIV_Byte		algoID;
	PIV_Byte		algoInputBuffer[BUFSIZ];
	PIV_ULong32		algoInputBufferLength = 0;
	PIV_Byte*		algoOutput = NULL;
	PIV_ULong32		algoOutputLength = 0;

	if ((NULL == pivRet) || (INVALID_CARD_HANDLE == cardHandle) || (NULL == ucDataToSign) || (0 == cbDataToSign) || (NULL == pcbSignature))
	{
		wprintf_s (L"SignData: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	// ****
	// Use of the RSA 2048 encoding algorithm with the PKCS #1 v1.5 padding sheme
	algoID = ALGORITHMID_RSA_2048;
	unPaddedAlgoInputLen = 0x100;

	// The data to sign exceed the maximum length allowed with this algorithm
	if (cbDataToSign > (unPaddedAlgoInputLen-11))
	{
		wprintf_s (L"SignData: Data to sign exceeds the maximum size allowed by the algorithm.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	pPaddedAlgoInput = (PIV_Byte*) malloc(unPaddedAlgoInputLen);
	if (NULL == pPaddedAlgoInput)
	{
		wprintf_s (L"SignData: allocation error.");

		dwRet = ERROR_OUTOFMEMORY;
		goto end_function;
	}

	/* Block type 0 */
	*pPaddedAlgoInput = 0;

	/* Block type 1 */
	*(pPaddedAlgoInput+1) = 1;

	/* Padding with 0xFF value */
	memset (pPaddedAlgoInput+2, 0xff, unPaddedAlgoInputLen - 2 - 1 - cbDataToSign);

	/* Separator */
	pPaddedAlgoInput[unPaddedAlgoInputLen - 1 - cbDataToSign] = 0;

	/* Data to sign */
	memcpy(pPaddedAlgoInput + (unPaddedAlgoInputLen - cbDataToSign), ucDataToSign, cbDataToSign);

	// Format the input buffer
	algoInputBuffer[0] = 0x7C; // Dynamic Authentication Template
	algoInputBuffer[1] = 0x82;
	algoInputBuffer[2] = (PIV_Byte)(((unPaddedAlgoInputLen+6) & 0xff00)>>8);
	algoInputBuffer[3] = (PIV_Byte)((unPaddedAlgoInputLen+6) & 0x00ff);
	algoInputBuffer[4] = BER_TAG_RESPONSE;
	algoInputBuffer[5] = 0x00;
	algoInputBuffer[6] = BER_TAG_CHALLENGE;
	algoInputBuffer[7] = 0x82;
	algoInputBuffer[8] = (PIV_Byte)((unPaddedAlgoInputLen & 0xff00)>>8);
	algoInputBuffer[9] = (PIV_Byte)(unPaddedAlgoInputLen & 0x00ff);
	memcpy(algoInputBuffer+10,pPaddedAlgoInput,unPaddedAlgoInputLen);
	algoInputBufferLength = unPaddedAlgoInputLen + 10;

	algoOutputLength = algoInputBufferLength ;

	if ((*pcbSignature) < algoOutputLength)
	{
		*pcbSignature = algoOutputLength;
		dwRet = ERROR_INSUFFICIENT_BUFFER;

		goto end_function;
	}

	algoOutput = (PIV_Byte*) malloc(algoOutputLength);
	if (NULL == algoOutput)
	{
		wprintf_s (L"SignData: allocation error.");

		dwRet = ERROR_OUTOFMEMORY;
		goto end_function;
	}

	// Initializing with 0xFF value
	memset (algoOutput, 0xff, algoOutputLength);

	// Call piv function
	*pivRet = pivCrypt(cardHandle,
		algoID,
		keyReference,
		algoInputBuffer,
		algoInputBufferLength,
		algoOutput,
		&algoOutputLength);

	if (PIV_OK != (*pivRet))
	{
		dwRet = ERROR_PIV_API;

		goto end_function;
	}

	if (NULL != ucSignature)
	{
		memcpy_s(ucSignature, *pcbSignature, algoOutput, algoOutputLength);
	}

	*pcbSignature = algoOutputLength;

end_function:
	if (NULL != pPaddedAlgoInput)
	{
		free(pPaddedAlgoInput);
	}

	if (NULL != algoOutput)
	{
		free(algoOutput);
	}

	return dwRet ;
}


DWORD ListObjects(const PIV_CARDHANDLE cardHandle, OBJECT_INDEX* pObjects, PDWORD pcbObjects)
{
	DWORD			dwRet = ERROR_SUCCESS;
	LPSTR			pObjectOID = NULL;
	OBJECT_INDEX	nObjectIndex = OBJECT_INDEX_NO_OBJECT;
	OBJECT_INDEX	objects[OBJECT_INDEX_CARDHOLDER_IRIS_IMAGES+1];
	DWORD			wMenuIndex = 0;
	int				i = 0;
	PIV_RV			pivRet	= PIV_OK;

	// Parameters
	PIV_Byte		oid[BUFSIZ];
	PIV_ULong32		oidLength = 0;
	PIV_ULong32		dataLength = 0;

	if ((INVALID_CARD_HANDLE == cardHandle) || (NULL == pcbObjects))
	{
		wprintf_s (L"ListObjects: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	// Initialize array to NO_OBJECT
	for (i=0; i<_countof(objects); i++)
	{
		objects[i] = OBJECT_INDEX_NO_OBJECT;
	}

	for (nObjectIndex=OBJECT_INDEX_CARD_COMPABILITY_CONTAINER; nObjectIndex<=OBJECT_INDEX_CARDHOLDER_IRIS_IMAGES; nObjectIndex++)
	{
		pObjectOID = g_objectInfoTable[nObjectIndex].szObjectOID;

		memcpy_s(oid, sizeof(oid), pObjectOID, strlen(pObjectOID));
		oidLength = (PIV_ULong32)strlen(pObjectOID);
		dataLength = 0;

		pivRet = pivGetData(
			cardHandle,
			oid,
			oidLength,
			NULL,
			&dataLength);

		if (((PIV_INSUFFICIENT_BUFFER == pivRet) && (dataLength>0)) || (PIV_SECURITY_CONDITIONS_NOT_SATISFIED == pivRet))
		{
			// Add available and initialized piv objects to the array
			objects[wMenuIndex++] = nObjectIndex;
		}
	}

	if (0 == wMenuIndex)
	{
		*pcbObjects = wMenuIndex;
		goto end_function;
	}

	if ((*pcbObjects) < wMenuIndex)
	{
		*pcbObjects = wMenuIndex;
		dwRet = ERROR_INSUFFICIENT_BUFFER;
		goto end_function;
	}

	if (NULL != pObjects)
	{
		// Return the piv objects array
		memcpy_s(pObjects, (*pcbObjects)*sizeof(OBJECT_INDEX), objects, wMenuIndex*sizeof(OBJECT_INDEX));
	}

	// Return the number of elements
	*pcbObjects = wMenuIndex;

end_function:
	return dwRet;
}


DWORD ReadObject(PIV_RV* pivRet, const PIV_CARDHANDLE cardHandle, const OBJECT_INDEX nObjectIndex, LPBYTE ucObjectData, PDWORD pcbObjectData)
{
	DWORD			dwRet = ERROR_SUCCESS;
	LPSTR			pLabel = NULL;

	// Parameters
	LPSTR			oid = NULL;
	PIV_Byte*		data = NULL;
	PIV_ULong32		dataLength = 0;

	if ((NULL == pivRet) || (INVALID_CARD_HANDLE == cardHandle) || (OBJECT_INDEX_NO_OBJECT == nObjectIndex) || (NULL == pcbObjectData))
	{
		wprintf_s (L"ReadObject: bad parameter.");

		dwRet = ERROR_INVALID_PARAMETER;
		goto end_function;
	}

	oid = g_objectInfoTable[nObjectIndex].szObjectOID;

	pLabel = g_objectInfoTable[nObjectIndex].szObjectLabel;

	if ((NULL != ucObjectData) && ((*pcbObjectData) > 0))
	{
		dataLength = *pcbObjectData;
		data = (PIV_Byte*) malloc(dataLength);
	}

	*pivRet = pivGetData(cardHandle,
		(PIV_Byte*)oid,
		(PIV_ULong32)strlen(oid),
		data,
		&dataLength);

	if (PIV_OK != (*pivRet))
	{
		if (PIV_INSUFFICIENT_BUFFER == (*pivRet))
		{
			*pcbObjectData = dataLength;
			dwRet = ERROR_INSUFFICIENT_BUFFER;
		}
		else
		{
			dwRet = ERROR_PIV_API;
		}

		goto end_function;
	}

	if (NULL != ucObjectData)
	{
		memcpy_s(ucObjectData, *pcbObjectData, data, dataLength);
	}

	*pcbObjectData = dataLength;

end_function:
	if (NULL != data)
	{
		free(data);
	}

	return dwRet;
}
