/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */

package piv;

import ui.MainWindow;
import ui.ScenarioInterface;
import ui.MainWindow.LogType;
import interfaces.PIVReturnCodes;
import classes.PIVAPIException;
import classes.PivApiWrapper;


public class Scenarios implements ScenarioInterface
{
	/*
	 * This field contains the identifier of the algorithm used to sign data in this sample.
	 * It is set to 0x07, which represents the RSA 2048 encoding algorithm with the 
	 * PKCS #1 v1.5 padding scheme. In a production environment, it should depend on the certificate 
	 * found on the smart card. The "right" way to determine which algorithm to choose would be to 
	 * parse the public key of the certificate, deduce the type of the certificate, and choose the 
	 * algorithm accordingly. However, this process goes beyond the scope of this sample. Thus, only 
	 * 2048 bits RSA certificates are supported by this application.
	 */
	private final byte m_algoID = (byte)0x07;
	
	private long m_hCard = -1;
	
	//
	// According to RFC 3447 described here: http://www.ietf.org/rfc/rfc3447.txt (section 9.2 EMSA-PKCS1-v1_5), the data to be signed must be formatted as follows:
	// - Padding
	// - Hash algorithm OID
	// - Message hash
	// Original message: Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.
	// MD5 algo is used for signature sample below:
	// - Padding: 00 01 FF  FF 00
	// - Hash algorithm OID: 30 20 30 0c 06 08 2a 86 48 86 f7 0d 02 05 05 00 04 10
	// - Message hash: 81 8c 6e 60 1a 24 f7 27 50 da 0f 6c 9b 8e be 28
	private final byte[] m_dataToSign = {// Hash algorithm OID
										 (byte)0x30, (byte)0x20, (byte)0x30, (byte)0x0C, (byte)0x06, (byte)0x08, (byte)0x2A, 
										 (byte)0x86, (byte)0x48, (byte)0x86, (byte)0xF7, (byte)0x0D, (byte)0x02, (byte)0x05, 
										 (byte)0x05, (byte)0x00, (byte)0x04, (byte)0x10,
										 // Message hash
										 (byte)0x81, (byte)0x8c, (byte)0x6e, (byte)0x60, (byte)0x1a, (byte)0x24, (byte)0xf7, 
										 (byte)0x27, (byte)0x50, (byte)0xda, (byte)0x0f, (byte)0x6c, (byte)0x9b, (byte)0x8e, 
										 (byte)0xbe, (byte)0x28};
	
	// Number of PIV objects used by this sample.
	private int m_nbObjects = 14;
	// Hardcoded number of the different certificates used by this sample; only 
	// the following certificates are used: PIV Authentication, Digital Signature, Card Authentication.
	private int m_nbCertificates = 3;
	
	// The lists containing hardcoded properties of PIV objects.
	private String [] m_listLabels = new String[m_nbObjects];
	private String [] m_listOIDs = new String[m_nbObjects];
	private boolean [] m_listPINNeed = new boolean[m_nbObjects];
	
	// List that will contain the list of smart card readers on the machine.
	private String [] m_readerList = null;
	
	// The lists containing hardcoded properties of the three certificates.
	private String [] m_certOnCurrentCardLabelsList = null;
	private String [] m_certOnCurrentCardOIDsList = null;
	private int [] m_certIndexList = new int[m_nbCertificates];
	private boolean [] m_certPresence = new boolean[m_nbCertificates];
	private int m_cardCertsCnt = 0;
	private byte [] m_keyReferenceList = new byte[m_nbCertificates];
	private int [] m_certsAuthentNb = new int[m_nbCertificates];
	
	// Lists that will contain the PIV objects present on the card. 
	private String [] m_objOnCurrentCardLabelsList = null;
	private String [] m_objOnCurrentCardOIDsList = null;
	private boolean [] m_objPresence = new boolean[m_nbObjects];
	private int m_cardObjCnt = 0;
	
	// The index of the selected PIV object.
	private int m_selectedObject = 0;
	
	private MainWindow m_mainWindow;
	
	// The wrapper object
	private PivApiWrapper myPivApi = null;	
	
	public Scenarios()
	{
		m_mainWindow = new MainWindow(this, new String[]{"PIV signature", "PIV objects read"}, "HID Global PIV API Java sample");
		
		// Instantiate the PIV API Wrapper class.
		try
		{
			myPivApi = new PivApiWrapper();
			
		    if (myPivApi == null)
		    {
		    	m_mainWindow.print("myPivApi is null", LogType.ScenarioError);
            }
		}
        catch (Exception ex)
		{
        	m_mainWindow.print("Exception : " + ex.toString(), LogType.ScenarioError);
            System.exit(-1);
        }
		
		m_mainWindow.print("------------------------------------------------------------------------------------");
    	m_mainWindow.print("HID Global PIV API Java Sample");
    	m_mainWindow.print("Copyright  2016 HID Global Corporation/ASSA ABLOY AB. All rights reserved.");
    	m_mainWindow.print("------------------------------------------------------------------------------------");
		
		String apiVersion = null;
		try
		{
			apiVersion = myPivApi.pivMiddlewareVersion();
		}
        catch (Exception ex)
		{
        	m_mainWindow.print("pivMiddlewareVersion failed with error " + ex.toString(), LogType.ScenarioError);
            System.exit(-1);
        }
		String popupMsg = 		"This sample demonstrates how to use the PIV API version \""+apiVersion+"\" provided by ActivClient 7.1 from a Java program.\n"
							+	"It implements following scenarios:\n"
							+	"        -    How to use a PIV signature key to sign some data.\n"
							+	"        -    How to read PIV objects from the card.\n";
		m_mainWindow.showMessageDialog(popupMsg);

        // Fill the lists with PIV objects properties : 
        // 		- a friendly name
        //		- the OID of the object
        //		- whether it is protected by PIN
        m_listLabels[0] = new String("Card Capability Container");
        m_listOIDs	[0] = new String("2.16.840.1.101.3.7.1.219.0");
        m_listPINNeed[0] = false;
        
        m_listLabels[1] = new String("Card Holder Unique Identifier");
        m_listOIDs	[1] = new String("2.16.840.1.101.3.7.2.48.0");
        m_listPINNeed[1] = false;
        
        m_listLabels[2] = new String("X.509 Certificate for PIV Authentication");
        m_listOIDs	[2] = new String("2.16.840.1.101.3.7.2.1.1");
        m_listPINNeed[2] = false;
        // For the certificates, we need additional information. m_certIndexList contains the index in the global lists,
        // m_keyReferenceList contains the key reference for the certificate, and m_certsAuthentNb contains the type
        // of PIN authentication : 0 for no authentication, 1 for a single authentication for multiple signatures, and
        // 2 for an authentication for each signature.
        m_certIndexList[0] = 2;
        m_keyReferenceList[0] = (byte)0x9A;
        m_certsAuthentNb[0] = 1;
        
        m_listLabels[3] = new String("Card Holder Fingerprints");
        m_listOIDs	[3] = new String("2.16.840.1.101.3.7.2.96.16");
        m_listPINNeed[3] = true;
        
        m_listLabels[4] = new String("Security Object");
        m_listOIDs	[4] = new String("2.16.840.1.101.3.7.2.144.0");
        m_listPINNeed[4] = false;
        
        m_listLabels[5] = new String("Card Holder Facial Image");
        m_listOIDs	[5] = new String("2.16.840.1.101.3.7.2.96.48");
        m_listPINNeed[5] = true;
        
        m_listLabels[6] = new String("Printed Information");
        m_listOIDs	[6] = new String("2.16.840.1.101.3.7.2.48.1");
        m_listPINNeed[6] = true;
        
        m_listLabels[7] = new String("X.509 Certificate for Digital Signature");
        m_listOIDs	[7] = new String("2.16.840.1.101.3.7.2.1.0");
        m_listPINNeed[7] = false;
        m_certIndexList[1] = 7;
        m_keyReferenceList[1] = (byte)0x9C;
        m_certsAuthentNb[1] = 2;
        
        m_listLabels[8] = new String("X.509 Certificate for Key Management");
        m_listOIDs	[8] = new String("2.16.840.1.101.3.7.2.1.2");
        m_listPINNeed[8] = false;
        
        m_listLabels[9] = new String("X.509 Certificate for Card Authentication");
        m_listOIDs	[9] = new String("2.16.840.1.101.3.7.2.5.0");
        m_listPINNeed[9] = false;
        m_certIndexList[2] = 9;
        m_keyReferenceList[2] = (byte)0x9E;
        m_certsAuthentNb[2] = 0;
        
        m_listLabels[10] = new String("Discovery Object");
        m_listOIDs	[10] = new String("2.16.840.1.101.3.7.2.96.80");
        m_listPINNeed[10] = false;
        
        m_listLabels[11] = new String("Key History Object");
        m_listOIDs	[11] = new String("2.16.840.1.101.3.7.2.96.96");
        m_listPINNeed[11] = false;
        
        m_listLabels[12] = new String("Retired X.509 Certificate for Key Management 1");
        m_listOIDs	[12] = new String("2.16.840.1.101.3.7.2.16.1");
        m_listPINNeed[12] = false;
        
        m_listLabels[13] = new String("Cardholder Iris Image");
        m_listOIDs	[13] = new String("2.16.840.1.101.3.7.2.16.21");
        m_listPINNeed[13] = true;
	}
	
	/**
    * Executes the steps of the PIV signature scenario.
    *
    */
	public void pivSignature()
	{
		m_mainWindow.print("Starting PIV signature scenario...", LogType.ScenarioBoundary);
		if(!preamble())
		{
			m_mainWindow.print("PIV signature scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!getSelectedCertificate())
		{
			postamble(false);
			m_mainWindow.print("PIV signature scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		switch(m_certsAuthentNb[m_selectedObject])
		{
		case 0:
			signData(m_keyReferenceList[m_selectedObject], m_algoID);
			signData(m_keyReferenceList[m_selectedObject], m_algoID);
			postamble(false);
			break;
		case 1:
			if(!logIntoCardApplication())
			{
				postamble(false);
				m_mainWindow.print("PIV signature scenario aborted!", LogType.ScenarioBoundary);
				return;
			}
			signData(m_keyReferenceList[m_selectedObject], m_algoID);
			signData(m_keyReferenceList[m_selectedObject], m_algoID);
			postamble(true);
			break;
		case 2:
			if(!logIntoCardApplication())
			{
				postamble(false);
				m_mainWindow.print("PIV signature scenario aborted!", LogType.ScenarioBoundary);
				return;
			}
			signData(m_keyReferenceList[m_selectedObject], m_algoID);
			if(!logIntoCardApplication())
			{
				postamble(false);
				m_mainWindow.print("PIV signature scenario aborted!", LogType.ScenarioBoundary);
				return;
			}
			signData(m_keyReferenceList[m_selectedObject], m_algoID);
			postamble(true);
			break;	
		default:
			postamble(false);
			break;
		}
		// Once the data is signed, we just display it. The signature is not verified : 
		// this goes beyond the scope of this sample.
		m_mainWindow.print("PIV signature scenario completed.", LogType.ScenarioBoundary);
	}
	
	/**
    * Executes the steps of the PIV Read objects scenario.
    *
    */
	public void pivObjectsRead()
	{
		m_mainWindow.print("Starting PIV objects read scenario...", LogType.ScenarioBoundary);
		if(!preamble())
		{
			postamble(false);
			m_mainWindow.print("PIV objects read scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(!getSelectedObject())
		{
			postamble(false);
			m_mainWindow.print("PIV objects read scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		if(m_listPINNeed[m_selectedObject] && !logIntoCardApplication())
		{
			postamble(false);
			m_mainWindow.print("PIV objects read scenario aborted!", LogType.ScenarioBoundary);
			return;
		}
		readObject();
		postamble(m_listPINNeed[m_selectedObject]);
		m_mainWindow.print("PIV objects read scenario completed.", LogType.ScenarioBoundary);
	}
	
	/**
    * Ends a scenario by terminating the PIV connection : log out of the card if needed, 
    * then disconnect from the reader.
    *
    */
	public void postamble(boolean loggedIn)
	{
		if(loggedIn)
		{
			logoutOfCardApplication();
		}
		disconnect();
	}
	

	/**
    * Gets the list of smart card readers (by calling getReaderList), then
    * prompts the user with a popup letting him choose the reader to use.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean preamble()
	{
		listReaders();
		if(m_readerList == null || m_readerList.length == 0)
		{
			m_mainWindow.print("No card reader detected.");
			return false;
		}
		int choice = m_mainWindow.showSelectionDialog("Select the reader",  m_readerList);
		if(choice == -1 || !connect(m_readerList[choice]) || !selectCardApplication())
		{
			return false;
		}
		return true;
	}
	
	/**
    * Reads on the card to get the list of certificates (by calling checkCertificates), then
    * prompts the user with a popup letting him choose the certificate that will be used to sign.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean getSelectedCertificate()
	{
		listCertificates();
		if(m_cardCertsCnt == 0)
		{
			m_mainWindow.print("No certificate found on the card.");
			return false;
		}
		m_certOnCurrentCardLabelsList = new String[m_cardCertsCnt];
		m_certOnCurrentCardOIDsList = new String[m_cardCertsCnt];
		int[] objIdx = new int[m_cardCertsCnt];
		int i = 0, cnt = 0;
		while(i < m_nbCertificates)
		{
			if(m_certPresence[i])
			{
				m_certOnCurrentCardLabelsList[cnt] = m_listLabels[m_certIndexList[i]];
				m_certOnCurrentCardOIDsList[cnt] = m_listOIDs[m_certIndexList[i]];
				objIdx[cnt] = i;
				cnt++;
			}
			i++;
		}
		int choice = m_mainWindow.showSelectionDialog("Select the certificate",  m_certOnCurrentCardLabelsList);
		if(choice == -1)
		{
			return false;
		}
		m_selectedObject = objIdx[choice];
		return true;
	}
	
	/**
    * Reads on the card to get the list of objects (by calling checkObjects), then
    * prompts the user with a popup letting him choose the object to be printed.
    *
    * @return boolean 		false if the operation failed or was canceled, true otherwise
    */
	public boolean getSelectedObject()
	{
		listObjects();
		if(m_cardObjCnt == 0)
		{
			m_mainWindow.print("No object found on the card.");
			return false;
		}
		m_objOnCurrentCardLabelsList = new String[m_cardObjCnt];
		m_objOnCurrentCardOIDsList = new String[m_cardObjCnt];
		int[] objIdx = new int[m_cardObjCnt];
		int i = 0, cnt = 0;
		while(i < m_nbObjects)
		{
			if(m_objPresence[i])
			{
				m_objOnCurrentCardLabelsList[cnt] = m_listLabels[i];
				m_objOnCurrentCardOIDsList[cnt] = m_listOIDs[i];
				objIdx[cnt] = i;
				cnt++;
			}
			i++;
		}
		int choice = m_mainWindow.showSelectionDialog("Select the object",  m_objOnCurrentCardLabelsList);
		if(choice == -1)
		{
			return false;
		}
		m_selectedObject = objIdx[choice];
		return true;
	}
	
	/**
    * Prints the contents of the currently selected PIV object to the text area.
    * Calls pivGetData in order to get the contents of the object.
    *
    */
	public void readObject()
	{
		String sOID = m_listOIDs[m_selectedObject];
		byte [] data;
		try 
		{
			data = myPivApi.pivGetData(m_hCard, sOID);
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivGetData failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
			if(m_listPINNeed[m_selectedObject])
			{
				logoutOfCardApplication();
			}
			return;
		}
		
		String sdata = Helper.byteArrayToASCIIString(data);
		m_mainWindow.print(m_listLabels[m_selectedObject] + " = ");
		m_mainWindow.print(sdata, LogType.APIInfo);
		return;
	}
	
	/**
    * Gets the list of available smart card readers by calling pivConnect.
    *
    */
	public void listReaders()
	{
		// Retrieves the reader list. See NIST.SP.800-73-4
		byte[] ConnectionDescription = new byte[1000];
		ConnectionDescription[0] = (byte)0x7f;
		ConnectionDescription[1] = (byte)0x21;
		ConnectionDescription[2] = (byte)0x04;
		ConnectionDescription[3] = (byte)0x81;
		ConnectionDescription[4] = (byte)0x00;
		ConnectionDescription[5] = (byte)0x90;
		ConnectionDescription[6] = (byte)0x00;
		
		try 
		{
			myPivApi.pivConnect(true, ConnectionDescription);
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivConnect failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
			return;
		}
		
		// Parse the connection string to retrieve all the readers.
        String strMsg = new String();
        int iNbReader = 0;
        
		for (int i = 0 ; i < ConnectionDescription.length ; i++)
        {
        	if ((char)ConnectionDescription[i] != '\0')
        		strMsg += (char)ConnectionDescription[i];
        	else
        	{
        		if (strMsg.length() == 0)
        			break;
        		else
        		{
        			strMsg = new String();
        			iNbReader++;
        		}
        	}
        }
		
		if (iNbReader > 0)
		{
			int iCounter = 0;
			m_readerList = new String[iNbReader];

			strMsg = new String();
			for (int i = 0 ; i < ConnectionDescription.length ; i++)
	        {
	        	if ((char)ConnectionDescription[i] != '\0')
	        		strMsg += (char)ConnectionDescription[i];
	        	else
	        	{
	        		if (strMsg.length() == 0)
	        			break;
	        		else
	        		{
	        			m_readerList[iCounter] = strMsg;
        				iCounter++;
	        			
        				strMsg = new String();
	        		}
	        	}
	        }
		}
	}
	
	/**
    * Connect to the selected card reader by calling pivConnect.
    *
    * @param reader		the name of the card reader to connect to
    */
	public boolean connect(String reader)
	{
		// Build the connection string.
		int unConnectionDescriptionLen = 7 + reader.length();
		
		byte[] ConnectionDescription = new byte[unConnectionDescriptionLen];
		ConnectionDescription[0] = (byte)0x7f;
		ConnectionDescription[1] = (byte)0x21;
		ConnectionDescription[2] = (byte)(unConnectionDescriptionLen-3);
		ConnectionDescription[3] = (byte)0x81;
		ConnectionDescription[4] = (byte)reader.length();
		
		for (int i = 0 ; i < reader.length() ; i++)
        {
			ConnectionDescription[5+i] = (byte)reader.charAt(i);
        }
		
		ConnectionDescription[unConnectionDescriptionLen-2] = (byte)0x90;
		ConnectionDescription[unConnectionDescriptionLen-1] = (byte)0x00;

		try 
		{
			m_hCard = myPivApi.pivConnect(true, ConnectionDescription);
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivConnect failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
			return false;
		}
		return true;
	}

	/**
    * Logs out of the card by calling pivLogoutOfCardApplication
    *
    */
	public void logoutOfCardApplication()
	{
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		try 
		{
			myPivApi.pivLogoutOfCardApplication(m_hCard);
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivLogoutOfCardApplication failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
		}
	}
	
	/**
    * Disconnects from the selected card reader by calling pivDisconnect.
    *
    */
	public void disconnect()
	{
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		try 
		{
			myPivApi.pivDisconnect(m_hCard);
			m_hCard = -1;
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivDisconnect failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
		}
	}
	
	/**
    * Uses the selected certificate to sign some data
    *
    * @param KeyReference	the reference key of the certificate
    * @param algorithmIdentifier	the identifier of the algorithm used to sign
    */
	public void signData(byte KeyReference, byte algorithmIdentifier)
    {
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		//	Get the size of the input buffer according to the algoId.
		int nPaddedAlgoInputLen = 0x00;
		switch(algorithmIdentifier)
		{
		case 0x00:
		case 0x06:
			nPaddedAlgoInputLen = 0x80;
			break;

		case 0x07:
			nPaddedAlgoInputLen = 0x100;
			break;

		case 0x05:
			nPaddedAlgoInputLen = 0x180;
			break;
		}
					
		if (nPaddedAlgoInputLen == 0)
		{
			m_mainWindow.print("Invalid algorithmIdentifier", LogType.ScenarioError);
			return;
		}
		
		int TextLen = m_dataToSign.length;
		
		//	Create the padded byte array
		byte [] pPaddedAlgoInput = new byte[nPaddedAlgoInputLen];
		pPaddedAlgoInput[0] = 0;
		pPaddedAlgoInput[1] = 1;
		for (int i=2; i<nPaddedAlgoInputLen  - 1 - TextLen; i++)
		{
			pPaddedAlgoInput[i] = (byte)0xFF;
		}
		
		pPaddedAlgoInput[nPaddedAlgoInputLen-TextLen-1] = (byte)0;
		
		for (int i=0; i<TextLen; i++)
		{
			pPaddedAlgoInput[i+nPaddedAlgoInputLen-TextLen] = m_dataToSign[i];
		}
		
		//	Create the buffer formatted as pivCrypt() expects.
		int unAlgoInputBufferLen = nPaddedAlgoInputLen;
		if (nPaddedAlgoInputLen+4 <= 0x7f)
		{
			unAlgoInputBufferLen += 6;
		}
		else if (nPaddedAlgoInputLen+4 > 0x7f && nPaddedAlgoInputLen+5 < 0x100)
		{
			unAlgoInputBufferLen += 8;
		}
		else if (nPaddedAlgoInputLen+5 >= 0x100)
		{
			unAlgoInputBufferLen += 10;
		}
		
		byte [] usAlgoInputBuffer = new byte[unAlgoInputBufferLen];
		
		usAlgoInputBuffer[0] = (byte)0x7C; // Dynamic Authentication Template
		
		if (nPaddedAlgoInputLen+4 <= 0x7f)
		{
			usAlgoInputBuffer[1] = (byte)(nPaddedAlgoInputLen + 4);
			usAlgoInputBuffer[2] = (byte)0x82; // Response
			usAlgoInputBuffer[3] = (byte)0x00;
			usAlgoInputBuffer[4] = (byte)0x81; // Challenge
			usAlgoInputBuffer[5] = (byte)nPaddedAlgoInputLen;

			Helper.copyArrays(usAlgoInputBuffer, 6, pPaddedAlgoInput);
		}
		else if (nPaddedAlgoInputLen+4 > 0x7f && nPaddedAlgoInputLen+5 < 0x100)
		{
			usAlgoInputBuffer[1] = (byte)0x81;
			usAlgoInputBuffer[2] = (byte)(nPaddedAlgoInputLen + 5);
			usAlgoInputBuffer[3] = (byte)0x82; // Response
			usAlgoInputBuffer[4] = (byte)0x00;
			usAlgoInputBuffer[5] = (byte)0x81; // Challenge
			usAlgoInputBuffer[6] = (byte)0x81;
			usAlgoInputBuffer[7] = (byte)nPaddedAlgoInputLen;

			Helper.copyArrays(usAlgoInputBuffer, 8, pPaddedAlgoInput);
		}
		else if (nPaddedAlgoInputLen+5 >= 0x100)
		{
			usAlgoInputBuffer[1] = (byte)0x82;
			usAlgoInputBuffer[2] = (byte)(((nPaddedAlgoInputLen+6) & 0xff00) >> 8);
			usAlgoInputBuffer[3] = (byte)((nPaddedAlgoInputLen+6) & 0x00ff);
			usAlgoInputBuffer[4] = (byte)0x82; // Response
			usAlgoInputBuffer[5] = (byte)0x00;
			usAlgoInputBuffer[6] = (byte)0x81; // Challenge
			usAlgoInputBuffer[7] = (byte)0x82;
			usAlgoInputBuffer[8] = (byte)((nPaddedAlgoInputLen & 0xff00)>>8);
			usAlgoInputBuffer[9] = (byte)(nPaddedAlgoInputLen & 0x00ff);

			Helper.copyArrays(usAlgoInputBuffer, 10, pPaddedAlgoInput);
		}
		byte [] data;
		try
		{
			data = myPivApi.pivCrypt(m_hCard, algorithmIdentifier, KeyReference, usAlgoInputBuffer);				
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivCrypt failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
			return;
		}
		
		m_mainWindow.print("signed data:");
		m_mainWindow.print(Helper.byteArrayToASCIIString(data), LogType.APIInfo);
    }
	
	/**
    * Fills the list of certificates present on the card.
    *
    */
	public void listCertificates()
	{
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		m_cardCertsCnt = 0;
		for(int i = 0; i < m_nbCertificates; i++)
		{
			String sOID = m_listOIDs[m_certIndexList[i]];
			try 
			{
				if(myPivApi.pivGetData(m_hCard, sOID).length > 0)
				{
					m_certPresence[i] = true;
					m_cardCertsCnt++;
				}
			}
			catch(PIVAPIException ex)
			{
				m_certPresence[i] = false;
				if(ex.getErrorCode() == PIVReturnCodes.PIV_SECURITY_CONDITIONS_NOT_SATISFIED)
				{
					m_certPresence[i] = true;
					m_cardCertsCnt++;
				}
			}
		}
	}
	
	/**
    * Fills the list of objects present on the card.
    *
    */
	public void listObjects()
	{
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return;
		}
		m_cardObjCnt = 0;
		for(int i = 0; i < m_nbObjects; i++)
		{
			String sOID = m_listOIDs[i];
			try 
			{
				if(myPivApi.pivGetData(m_hCard, sOID).length > 0)
				{
					m_objPresence[i] = true;
					m_cardObjCnt++;					
				}
				else
				{
					m_objPresence[i] = false;
				}
			}
			catch(PIVAPIException ex)
			{
				m_objPresence[i] = false;
				if(ex.getErrorCode() == PIVReturnCodes.PIV_SECURITY_CONDITIONS_NOT_SATISFIED)
				{
					m_objPresence[i] = true;
					m_cardObjCnt++;
				}
			}
		}
	}
	
	/**
    * Calls pivSelectCardApplication to select the PIV application on the card.
    *
    * @return boolean 	false if the call failed, true otherwise.
    */
	public boolean selectCardApplication()
	{
		String sDefaultAID;
		byte [] ApplicationAID;
		
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return false;
		}
		sDefaultAID = "A00000030800001000";
		ApplicationAID = Helper.ConvAsciiToBin(sDefaultAID);
		
		try 
		{
			myPivApi.pivSelectCardApplication(m_hCard, ApplicationAID);
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivSelectCardApplication failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
			return false;
		}
		return true;
	}
	
	/**
    * Login to the card. This method asks the user for the PIN of the card, 
    * then calls pivLogIntoCardApplication to authenticate him.
    *
    * @return boolean	false if the authentication failed, or the user canceled. 
    * 					If this method returns true, the caller must log out of the card
    * 					when the operations are finished. 
    */
	public boolean logIntoCardApplication()
	{
		if (m_hCard == -1)
		{
			m_mainWindow.print("Invalid card handle", LogType.ScenarioError);
			return false;
		}
		// Build the authenticator array (PIN only). Get the pin
        String sPin = m_mainWindow.showPINDialog();
        if (sPin == null)
        {
        	return false;
        }

		int unAuthenticatorsLen = 3 + 2 + sPin.length() + 3;
		byte [] authenticator = new byte[unAuthenticatorsLen];
		authenticator[0] = (byte)0x67;
		authenticator[1] = (byte)0x81;
		authenticator[2] = (byte)(unAuthenticatorsLen-3);
		authenticator[3] = (byte)0x83;
		authenticator[4] = (byte)1;
		authenticator[5] = (byte)0x80;
		authenticator[6] = (byte)0x81;
		authenticator[7] = (byte)sPin.length();
		
		for (int i = 0 ; i < sPin.length() ; i++)
        {
			authenticator[8+i] = (byte)sPin.charAt(i);
        }
		try 
		{
			myPivApi.pivLogIntoCardApplication(m_hCard, authenticator);
		}
		catch(PIVAPIException ex)
		{
			m_mainWindow.print("pivLogIntoCardApplication failed with error " + myPivApi.getStringErrorCode(ex.getErrorCode()), LogType.APIError);
			return false;
		}
		return true;
	}

	@Override
	public void runScenario(int scenarioIdx)
	{
		switch(scenarioIdx)
		{
		case 0:
			pivSignature();
			break;
		case 1:
			pivObjectsRead();
			break;
		default:
			m_mainWindow.print("Combo box state corrupt", LogType.ScenarioError);
			break;
		}
	}
}
